/***************************************************************************

	Incredible Technologies/Strata system
	(8-bit blitter variant)

    driver by Aaron Giles

    Games supported:
		* Strata Bowling
		* Wheel of Fortune
		* Golden Tee Golf
		* Golden Tee Golf II [3 sets]
		* Slick Shot
		* Arlington Horse Racing
		* Peggle [2 sets]
		* Hot Shots Tennis
		* Rim Rockin' Basketball [2 sets]
		* Ninja Clowns

	Similar games that ran on advanced (32-bit) graphics hardware:
		* Time Killers
		* Bloodstorm
		* Golden Tee 3D Golf
		* World Class Bowling
		* Golden Tee Golf '97
		* Golden Tee Golf '98
		* Golden Tee Golf '99
		* Golden Tee Golf 2K
		* Hard Yardage

	No known (good) dumps:
		* Dyno-Bop
		* Poker Dice
		* Neck & Neck
		* Driver's Edge
		* Pairs
		* Street Fighter II: The Movie

	Known issues:
		* The credits screen in Peggle shows for less than half the time
		  it should
		* Slick Shot has large sections of code related to the trackball
		  inputs NOPped out
		* Rim Rockin' Basketball should use an HD6309, but that core is
		  broken, so we're using the 6809 for now
		* Ninja Clowns main ROM dump claims it's bad

****************************************************************************

	Memory map

****************************************************************************

	========================================================================
	CPU #1 (6809 games)
	========================================================================
	Note that many games have the regions 0000-0FFF and 1000-1FFF swapped.
	Also, Golden Tee Golf II (V2.2) has most of the addresses in the
	0000-0FFF range swizzled randomly.
	========================================================================
	0100          W   xxxxxxxx    Unknown
	0120          W   xxxxxxxx    Sound data
	0140        R     xxxxxxxx    Input port
	0140          W   xxxxxxxx    GROM blitter bank select
	0160        R     xxxxxxxx    Input port
	0160          W   xx------    Display page select
	              W   x-------       (upper page select)
	              W   -x------       (unknown)
	0180        R     xxxxxxxx    Input port/DIP switches
	0180          W   xxxxxxxx    Video RAM upper data latch
	01A0          W   --------    NMI acknowledge (on some boards)
	01C0-01C1   R/W   xxxxxxxx    Blitter chip address high
	01C2-01C3   R/W   xxxxxxxx    Blitter chip address low
	01C4-01C5   R/W   ---xxxxx    Blitter chip flags
	            R/W   ---x----       (render transparent)
	            R/W   ----x---       (expand RLE compressed data)
	            R/W   -----x--       (flip vertically)
	            R/W   ------x-       (flip horizontally)
	            R/W   -------x       (shift data left 4 bits)
	01C6-01C7   R     x-------    Blitter chip status (1=busy)
	01C6-01C7     W   --------    Blitter chip trigger
	01C8-01C9   R/W   xxxxxxxx    Blitter chip source width
	01CA-01CB   R/W   xxxxxxxx    Blitter chip source height
	01CC-01CD   R/W   xxxxxxxx    Blitter chip source data mask
	01CE-01CF   R/W   xxxxxxxx    Blitter chip output port
	              W   --x-----       (main program ROM bank select)
	01D0-01D1   R/W   xxxxxxxx    Blitter chip X starting coordinate
	01D2-01D3   R/W   xxxxxxxx    Blitter chip maximum scanline count
	01D4-01D5   R/W   xxxxxxxx    Blitter chip X ending coordinate
	01D6-01D7   R/W   xxxxxxxx    Blitter chip initial scanline skip
	01D8-01DF   R     xxxxxxxx    Blitter chip input ports
	01E0          W   xxxxxxxx    Palette address latch
	01E2-01E3     W   --xxxxxx    Palette data latches
	1000-11FF   R/W   xxxxxxxx    TMS34061 register access
	1200-13FF   R/W   xxxxxxxx    TMS34061 XY addressing access
	1400-15FF   R/W   xxxxxxxx    TMS34061 register access
	1600-17FF   R/W   xxxxxxxx    TMS34061 direct access
	1800-19FF   R/W   xxxxxxxx    TMS34061 shift register read
	1A00-1BFF   R/W   xxxxxxxx    TMS34061 shift register write
	2000-3FFF   R/W   xxxxxxxx    RAM (battery-backed)
	4000-7FFF   R     xxxxxxxx    Banked program ROM
	8000-FFFF   R     xxxxxxxx    Fixed program ROM
	========================================================================
	Interrupts:
		NMI generated by VBLANK
		IRQ generated by TMS34061 scanline interrupt
		FIRQ generated by blitter when finished
	========================================================================

	========================================================================
	CPU #1 (68000 games)
	========================================================================
	000000-003FFF   R/W   xxxxxxxx xxxxxxxx   RAM (battery-backed)
	004000-03FFFF   R     xxxxxxxx xxxxxxxx   Main program ROM
	100080          W     xxxxxxxx --------   Sound data
	100100          R     xxxxxxxx --------   Input port
	100100            W   xxxxxxxx --------   GROM blitter bank select
	100180          R     xxxxxxxx --------   Input port
	100100            W   xx------ --------   Display page select
	                  W   x------- --------       (upper page select)
	                  W   -x------ --------       (unknown)
	100240            W   xxxxxxxx --------   Video RAM upper data latch
	100280          R     xxxxxxxx --------   Input port/DIP switches
	100280            W   -------- --------   Unknown
	100300-10031F   R/W   xxxxxxxx xxxxxxxx   Blitter chip (as above)
	100380            W   xxxxxxxx --------   Palette address latch
	1003a0            W   --xxxxxx --------   Palette data latches
	110000-1101FF   R/W   xxxxxxxx xxxxxxxx   TMS34061 register access
	110200-1103FF   R/W   xxxxxxxx xxxxxxxx   TMS34061 XY addressing access
	110400-1105FF   R/W   xxxxxxxx xxxxxxxx   TMS34061 register access
	110600-1107FF   R/W   xxxxxxxx xxxxxxxx   TMS34061 direct access
	110800-1109FF   R/W   xxxxxxxx xxxxxxxx   TMS34061 shift register read
	110A00-110BFF   R/W   xxxxxxxx xxxxxxxx   TMS34061 shift register write
	========================================================================
	Interrupts:
		INT 3 generated by VBLANK
		INT 2 generated by blitter when finished
		INT 1 generated by ???
	========================================================================

	========================================================================
	CPU #2
	========================================================================
	0000          W   xxxxxxxx    Unknown
	1000        R     xxxxxxxx    Sound command input
	2000-2003   R/W   xxxxxxxx    Yamaha chip I/O
	3000-37FF   R/W   xxxxxxxx    RAM
	4000        R/W   xxxxxxxx    OKI 6295 I/O
	5000-5003   R/W   xxxxxxxx    6521 PIA chip (on early YM3812 boards)
	5000-500F   R/W   xxxxxxxx    6522 VIA chip (on later YM3812 boards)
	8000-FFFF   R     xxxxxxxx    Program ROM
	========================================================================
	Interrupts:
		NMI not connected
		IRQ generated by write to sound command input
		FIRQ generated by Yamaha chip, or by 6522 VIA (if present)
	========================================================================

***************************************************************************/


#include "driver.h"
#include "cpu/m6809/m6809.h"
#include "machine/6821pia.h"
#include "machine/ticket.h"
#include "vidhrdw/generic.h"
#include "vidhrdw/tms34061.h"
#include <math.h>


#define FULL_LOGGING	0

#define CLOCK_8MHz		(8000000)
#define CLOCK_12MHz		(12000000)



/*************************************
 *
 *	External definitions
 *
 *************************************/

/* video driver data & functions */
extern UINT8 *itech8_grom_bank;
extern UINT8 *itech8_display_page;

int itech8_vh_start(void);
void itech8_vh_stop(void);
void itech8_vh_screenrefresh(struct osd_bitmap *bitmap, int full_refresh);

READ_HANDLER( itech8_tms34061_r );
READ_HANDLER( itech8_blitter_r );

WRITE_HANDLER( itech8_tms34061_w );
WRITE_HANDLER( itech8_palette_address_w );
WRITE_HANDLER( itech8_palette_data_w );
WRITE_HANDLER( itech8_blitter_w );



/*************************************
 *
 *	Static data
 *
 *************************************/

static UINT8 blitter_int;
static UINT8 tms34061_int;
static UINT8 periodic_int;

static data8_t sound_data;

static data8_t pia_porta_data;
static data8_t pia_portb_data;

static data8_t *via6522;
static data16_t via6522_timer_count[2];
static void *via6522_timer[2];
static data8_t via6522_int_state;

static data8_t *main_ram;
static size_t main_ram_size;



/*************************************
 *
 *	6821 PIA interface
 *
 *************************************/

static WRITE_HANDLER( pia_porta_out );
static WRITE_HANDLER( pia_portb_out );

static struct pia6821_interface pia_interface =
{
	0, 0, 0, 0, 0, 0,						/* PIA inputs: A, B, CA1, CB1, CA2, CB2 */
	pia_porta_out, pia_portb_out, 0, 0,		/* PIA outputs: A, B, CA2, CB2 */
	0, 0									/* PIA IRQs: A, B */
};



/*************************************
 *
 *	Interrupt handling
 *
 *************************************/

void itech8_update_interrupts(int periodic, int tms34061, int blitter)
{
	/* update the states */
	if (periodic != -1) periodic_int = periodic;
	if (tms34061 != -1) tms34061_int = tms34061;
	if (blitter != -1) blitter_int = blitter;

	/* handle the 6809 case */
	if ((Machine->drv->cpu[0].cpu_type & ~CPU_FLAGS_MASK) == CPU_M6809)
	{
		/* just modify lines that have changed */
		if (periodic != -1) cpu_set_nmi_line(0, periodic ? ASSERT_LINE : CLEAR_LINE);
		if (tms34061 != -1) cpu_set_irq_line(0, M6809_IRQ_LINE, tms34061 ? ASSERT_LINE : CLEAR_LINE);
		if (blitter != -1) cpu_set_irq_line(0, M6809_FIRQ_LINE, blitter ? ASSERT_LINE : CLEAR_LINE);
	}

	/* handle the 68000 case */
	else
	{
		int level = 0;

		/* determine which level is active */
		if (blitter_int) level = 2;
		if (periodic_int) level = 3;

		/* update it */
		if (level)
			cpu_set_irq_line(0, level, ASSERT_LINE);
		else
			cpu_set_irq_line(0, 7, CLEAR_LINE);
	}
}



/*************************************
 *
 *	Interrupt generation
 *
 *************************************/

static int generate_nmi(void)
{
	/* signal the NMI */
	itech8_update_interrupts(1, -1, -1);
	itech8_update_interrupts(0, -1, -1);

	if (FULL_LOGGING) logerror("------------ VBLANK (%d) --------------\n", cpu_getscanline());
	return ignore_interrupt();
}


static WRITE_HANDLER( nmi_ack_w )
{
/* doesn't seem to hold for every game (e.g., hstennis) */
/*	cpu_set_nmi_line(0, CLEAR_LINE);*/
}


static void generate_sound_irq(int state)
{
	cpu_set_irq_line(1, M6809_FIRQ_LINE, state ? ASSERT_LINE : CLEAR_LINE);
}



/*************************************
 *
 *	Machine initialization
 *
 *************************************/

static void init_machine(void)
{
	/* make sure bank 0 is selected */
	if ((Machine->drv->cpu[0].cpu_type & ~CPU_FLAGS_MASK) == CPU_M6809)
		cpu_setbank(1, &memory_region(REGION_CPU1)[0x4000]);

	/* reset the PIA (if used) */
	pia_unconfig();
	pia_config(0, PIA_STANDARD_ORDERING, &pia_interface);
	pia_reset();

	/* reset the VIA chip (if used) */
	via6522_timer_count[0] = via6522_timer_count[1] = 0;
	via6522_timer[0] = via6522_timer[1] = 0;
	via6522_int_state = 0;

	/* reset the ticket dispenser */
	ticket_dispenser_init(200, TICKET_MOTOR_ACTIVE_HIGH, TICKET_STATUS_ACTIVE_HIGH);
}



/*************************************
 *
 *	Bank switching
 *
 *************************************/

static WRITE_HANDLER( blitter_w )
{
	/* bit 0x20 on address 7 controls CPU banking */
	if (offset / 2 == 7)
		cpu_setbank(1, &memory_region(REGION_CPU1)[0x4000 + 0xc000 * ((data >> 5) & 1)]);

	/* the rest is handled by the video hardware */
	itech8_blitter_w(offset, data);
}


static WRITE_HANDLER( rimrockn_bank_w )
{
	/* banking is controlled here instead of by the blitter output */
	cpu_setbank(1, &memory_region(REGION_CPU1)[0x4000 + 0xc000 * (data & 3)]);
}



/*************************************
 *
 *	Input handling
 *
 *************************************/

static READ_HANDLER( special_port0_r )
{
	data8_t result = readinputport(0);
	result = (result & 0xfe) | (pia_portb_data & 0x01);
	return result;
}



/*************************************
 *
 *	6821 PIA handling
 *
 *************************************/

WRITE_HANDLER( pia_porta_out )
{
	logerror("PIA port A write = %02x\n", data);
	pia_porta_data = data;
}


WRITE_HANDLER( pia_portb_out )
{
	logerror("PIA port B write = %02x\n", data);

	/* bit 0 provides feedback to the main CPU */
	/* bit 4 controls the ticket dispenser */
	/* bit 5 controls the coin counter */
	/* bit 6 controls the diagnostic sound LED */
	pia_portb_data = data;
	ticket_dispenser_w(0, (data & 0x10) << 3);
	coin_counter_w(0, (data & 0x20) >> 5);
}


WRITE_HANDLER( ym2203_portb_out )
{
	logerror("YM2203 port B write = %02x\n", data);

	/* bit 0 provides feedback to the main CPU */
	/* bit 5 controls the coin counter */
	/* bit 6 controls the diagnostic sound LED */
	/* bit 7 controls the ticket dispenser */
	pia_portb_data = data;
	ticket_dispenser_w(0, data & 0x80);
	coin_counter_w(0, (data & 0x20) >> 5);
}



/*************************************
 *
 *	Sound communication
 *
 *************************************/

static void delayed_sound_data_w(int data)
{
	sound_data = data;
	cpu_set_irq_line(1, M6809_IRQ_LINE, ASSERT_LINE);
}


static WRITE_HANDLER( sound_data_w )
{
	timer_set(TIME_NOW, data, delayed_sound_data_w);
}


static WRITE_HANDLER( gtg2_sound_data_w )
{
	/* on the later GTG2 board, they swizzle the data lines */
	data = ((data & 0x80) >> 7) |
	       ((data & 0x5d) << 1) |
	       ((data & 0x20) >> 3) |
	       ((data & 0x02) << 5);
	timer_set(TIME_NOW, data, delayed_sound_data_w);
}


static READ_HANDLER( sound_data_r )
{
	cpu_set_irq_line(1, M6809_IRQ_LINE, CLEAR_LINE);
	return sound_data;
}



/*************************************
 *
 *	Sound 6522 VIA handling
 *
 *************************************/

INLINE void update_via_int(void)
{
	/* if interrupts are enabled and one is pending, set the line */
	if ((via6522[14] & 0x80) && (via6522_int_state & via6522[14]))
		cpu_set_irq_line(1, M6809_FIRQ_LINE, ASSERT_LINE);
	else
		cpu_set_irq_line(1, M6809_FIRQ_LINE, CLEAR_LINE);
}


static void via6522_timer_callback(int which)
{
	via6522_int_state |= 0x40 >> which;
	update_via_int();
}


static WRITE_HANDLER( via6522_w )
{
	/* update the data */
	via6522[offset] = data;

	/* switch off the offset */
	switch (offset)
	{
		case 0:		/* write to port B */
			pia_portb_out(0, data);
			break;

		case 5:		/* write into high order timer 1 */
			via6522_timer_count[0] = (via6522[5] << 8) | via6522[4];
			if (via6522_timer[0])
				timer_remove(via6522_timer[0]);
			via6522_timer[0] = timer_pulse(TIME_IN_HZ(CLOCK_8MHz/4) * (double)via6522_timer_count[0], 0, via6522_timer_callback);

			via6522_int_state &= ~0x40;
			update_via_int();
			break;

		case 13:	/* write interrupt flag register */
			via6522_int_state &= ~data;
			update_via_int();
			break;

		default:	/* log everything else */
			if (FULL_LOGGING) logerror("VIA write(%02x) = %02x\n", offset, data);
			break;
	}
}


static READ_HANDLER( via6522_r )
{
	int result = 0;

	/* switch off the offset */
	switch (offset)
	{
		case 4:		/* read low order timer 1 */
			via6522_int_state &= ~0x40;
			update_via_int();
			break;

		case 13:	/* interrupt flag register */
			result = via6522_int_state & 0x7f;
			if (via6522_int_state & via6522[14]) result |= 0x80;
			break;
	}

	if (FULL_LOGGING) logerror("VIA read(%02x) = %02x\n", offset, result);
	return result;
}



/*************************************
 *
 *	16-bit memory shunts
 *
 *************************************/

static READ16_HANDLER( blitter16_r )
{
	return (itech8_blitter_r(offset * 2 + 0) << 8) + itech8_blitter_r(offset * 2 + 1);
}


static READ16_HANDLER( tms34061_16_r )
{
	/* since multiple XY accesses can move the pointer multiple times, we have to */
	/* be careful to only perform one read per access here; fortunately, the low */
	/* bit doesn't matter in XY addressing mode */
	if ((offset & 0x700) == 0x100)
	{
		int result = itech8_tms34061_r(offset * 2);
		return (result << 8) | result;
	}
	else
		return (itech8_tms34061_r(offset * 2 + 0) << 8) + itech8_tms34061_r(offset * 2 + 1);
}


static WRITE16_HANDLER( sound_data16_w )
{
	if (ACCESSING_MSB)
		sound_data_w(0, data >> 8);
}


static WRITE16_HANDLER( grom_bank16_w )
{
	if (ACCESSING_MSB)
		*itech8_grom_bank = data >> 8;
}


static WRITE16_HANDLER( display_page16_w )
{
	if (ACCESSING_MSB)
		*itech8_display_page = ~data >> 8;
}


static WRITE16_HANDLER( tms34061_latch16_w )
{
	if (ACCESSING_MSB)
		tms34061_latch_w(0, data >> 8);
}


static WRITE16_HANDLER( blitter16_w )
{
	if (ACCESSING_MSB)
		itech8_blitter_w(offset * 2 + 0, data >> 8);
	if (ACCESSING_LSB)
		itech8_blitter_w(offset * 2 + 1, data);
}


static WRITE16_HANDLER( palette_addr16_w )
{
	if (ACCESSING_MSB)
		itech8_palette_address_w(0, data >> 8);
}


static WRITE16_HANDLER( palette_data16_w )
{
	if (ACCESSING_MSB)
		itech8_palette_data_w(0, data >> 8);
}


static WRITE16_HANDLER( tms34061_16_w )
{
	if (ACCESSING_MSB)
		itech8_tms34061_w(offset * 2 + 0, data >> 8);
	else if (ACCESSING_LSB)
		itech8_tms34061_w(offset * 2 + 1, data);
}



/*************************************
 *
 *	NVRAM read/write
 *
 *************************************/

static void nvram_handler(void *file, int read_or_write)
{
	int i;

	if (read_or_write)
		osd_fwrite(file, main_ram, main_ram_size);
	else if (file)
		osd_fread(file, main_ram, main_ram_size);
	else
		for (i = 0; i < main_ram_size; i++)
			main_ram[i] = rand();
}



/*************************************
 *
 *	Main CPU memory handlers
 *
 *************************************/

/*------ common layout with TMS34061 at 0000 ------*/
static MEMORY_READ_START( tmslo_readmem )
	{ 0x0000, 0x0fff, itech8_tms34061_r },
	{ 0x1140, 0x1140, special_port0_r },
	{ 0x1160, 0x1160, input_port_1_r },
	{ 0x1180, 0x1180, input_port_2_r },
	{ 0x11c0, 0x11d7, itech8_blitter_r },
	{ 0x11d8, 0x11d9, input_port_3_r },
	{ 0x11da, 0x11db, input_port_4_r },
	{ 0x11dc, 0x11dd, input_port_5_r },
	{ 0x11de, 0x11df, input_port_6_r },
	{ 0x2000, 0x3fff, MRA_RAM },
	{ 0x4000, 0xffff, MRA_BANK1 },
MEMORY_END


static MEMORY_WRITE_START( tmslo_writemem )
	{ 0x0000, 0x0fff, itech8_tms34061_w },
	{ 0x1100, 0x1100, MWA_NOP },
	{ 0x1120, 0x1120, sound_data_w },
	{ 0x1140, 0x1140, MWA_RAM, &itech8_grom_bank },
	{ 0x1160, 0x1160, MWA_RAM, &itech8_display_page },
	{ 0x1180, 0x1180, tms34061_latch_w },
	{ 0x11a0, 0x11a0, nmi_ack_w },
	{ 0x11c0, 0x11df, blitter_w },
	{ 0x11e0, 0x11e0, itech8_palette_address_w },
	{ 0x11e2, 0x11e3, itech8_palette_data_w },
	{ 0x2000, 0x3fff, MWA_RAM, &main_ram, &main_ram_size },
	{ 0x4000, 0xffff, MWA_ROM },
MEMORY_END


/*------ common layout with TMS34061 at 1000 ------*/
static MEMORY_READ_START( tmshi_readmem )
	{ 0x1000, 0x1fff, itech8_tms34061_r },
	{ 0x0140, 0x0140, special_port0_r },
	{ 0x0160, 0x0160, input_port_1_r },
	{ 0x0180, 0x0180, input_port_2_r },
	{ 0x01c0, 0x01d7, itech8_blitter_r },
	{ 0x01d8, 0x01d9, input_port_3_r },
	{ 0x01da, 0x01db, input_port_4_r },
	{ 0x01dc, 0x01dd, input_port_5_r },
	{ 0x01de, 0x01df, input_port_6_r },
	{ 0x2000, 0x3fff, MRA_RAM },
	{ 0x4000, 0xffff, MRA_BANK1 },
MEMORY_END


static MEMORY_WRITE_START( tmshi_writemem )
	{ 0x1000, 0x1fff, itech8_tms34061_w },
	{ 0x0100, 0x0100, MWA_NOP },
	{ 0x0120, 0x0120, sound_data_w },
	{ 0x0140, 0x0140, MWA_RAM, &itech8_grom_bank },
	{ 0x0160, 0x0160, MWA_RAM, &itech8_display_page },
	{ 0x0180, 0x0180, tms34061_latch_w },
	{ 0x01a0, 0x01a0, nmi_ack_w },
	{ 0x01c0, 0x01df, blitter_w },
	{ 0x01e0, 0x01e0, itech8_palette_address_w },
	{ 0x01e2, 0x01e3, itech8_palette_data_w },
	{ 0x2000, 0x3fff, MWA_RAM, &main_ram, &main_ram_size },
	{ 0x4000, 0xffff, MWA_ROM },
MEMORY_END


/*------ Golden Tee Golf II 1992 layout ------*/
static MEMORY_READ_START( gtg2_readmem )
	{ 0x1000, 0x1fff, itech8_tms34061_r },
	{ 0x0100, 0x0100, input_port_0_r },
	{ 0x0120, 0x0120, input_port_1_r },
	{ 0x0140, 0x0140, input_port_2_r },
	{ 0x0180, 0x0197, itech8_blitter_r },
	{ 0x0198, 0x0199, input_port_3_r },
	{ 0x019a, 0x019b, input_port_4_r },
	{ 0x019c, 0x019d, input_port_5_r },
	{ 0x019e, 0x019f, input_port_6_r },
	{ 0x2000, 0x3fff, MRA_RAM },
	{ 0x4000, 0xffff, MRA_BANK1 },
MEMORY_END


static MEMORY_WRITE_START( gtg2_writemem )
	{ 0x1000, 0x1fff, itech8_tms34061_w },
	{ 0x01c0, 0x01c0, gtg2_sound_data_w },
	{ 0x0160, 0x0160, MWA_RAM, &itech8_grom_bank },
	{ 0x0120, 0x0120, MWA_RAM, &itech8_display_page },
	{ 0x01e0, 0x01e0, tms34061_latch_w },
	{ 0x0100, 0x0100, nmi_ack_w },
	{ 0x0180, 0x019f, blitter_w },
	{ 0x0140, 0x0140, itech8_palette_address_w },
	{ 0x0142, 0x0143, itech8_palette_data_w },
	{ 0x2000, 0x3fff, MWA_RAM, &main_ram, &main_ram_size },
	{ 0x4000, 0xffff, MWA_ROM },
MEMORY_END


/*------ Ninja Clowns layout ------*/
static MEMORY_READ16_START( ninclown_readmem )
	{ 0x000000, 0x003fff, MRA16_RAM },
	{ 0x004000, 0x07ffff, MRA16_ROM },
	{ 0x100100, 0x100101, input_port_0_word_r },
	{ 0x100180, 0x100181, input_port_1_word_r },
	{ 0x100280, 0x100281, input_port_2_word_r },
	{ 0x100300, 0x10031f, blitter16_r },
	{ 0x110000, 0x110fff, tms34061_16_r },
MEMORY_END


static MEMORY_WRITE16_START( ninclown_writemem )
	{ 0x000000, 0x00007f, MWA16_RAM },
	{ 0x000080, 0x003fff, MWA16_RAM, (data16_t **)&main_ram, &main_ram_size },
	{ 0x004000, 0x07ffff, MWA16_ROM },
	{ 0x100080, 0x100081, sound_data16_w },
	{ 0x100100, 0x100101, grom_bank16_w, (data16_t **)&itech8_grom_bank },
	{ 0x100180, 0x100181, display_page16_w, (data16_t **)&itech8_display_page },
	{ 0x100240, 0x100241, tms34061_latch16_w },
	{ 0x100280, 0x100281, MWA16_NOP },
	{ 0x100300, 0x10031f, blitter16_w },
	{ 0x100380, 0x100381, palette_addr16_w },
	{ 0x1003a0, 0x1003a1, palette_data16_w },
	{ 0x110000, 0x110fff, tms34061_16_w },
MEMORY_END



/*************************************
 *
 *	Sound CPU memory handlers
 *
 *************************************/

/*------ YM2203-based sound board ------*/
static MEMORY_READ_START( sound2203_readmem )
	{ 0x1000, 0x1000, sound_data_r },
	{ 0x2000, 0x2000, YM2203_status_port_0_r },
	{ 0x2002, 0x2002, YM2203_status_port_0_r },
	{ 0x3000, 0x37ff, MRA_RAM },
	{ 0x4000, 0x4000, OKIM6295_status_0_r },
	{ 0x8000, 0xffff, MRA_ROM },
MEMORY_END


static MEMORY_WRITE_START( sound2203_writemem )
	{ 0x0000, 0x0000, MWA_NOP },
	{ 0x2000, 0x2000, YM2203_control_port_0_w },
	{ 0x2001, 0x2001, YM2203_write_port_0_w },
	{ 0x2002, 0x2002, YM2203_control_port_0_w },
	{ 0x2003, 0x2003, YM2203_write_port_0_w },
	{ 0x3000, 0x37ff, MWA_RAM },
	{ 0x4000, 0x4000, OKIM6295_data_0_w },
	{ 0x8000, 0xffff, MWA_ROM },
MEMORY_END


/*------ YM3812-based sound board ------*/
static MEMORY_READ_START( sound3812_readmem )
	{ 0x1000, 0x1000, sound_data_r },
	{ 0x2000, 0x2000, YM3812_status_port_0_r },
	{ 0x3000, 0x37ff, MRA_RAM },
	{ 0x4000, 0x4000, OKIM6295_status_0_r },
	{ 0x5000, 0x5003, pia_0_r },
	{ 0x8000, 0xffff, MRA_ROM },
MEMORY_END


static MEMORY_WRITE_START( sound3812_writemem )
	{ 0x0000, 0x0000, MWA_NOP },
	{ 0x2000, 0x2000, YM3812_control_port_0_w },
	{ 0x2001, 0x2001, YM3812_write_port_0_w },
	{ 0x3000, 0x37ff, MWA_RAM },
	{ 0x4000, 0x4000, OKIM6295_data_0_w },
	{ 0x5000, 0x5003, pia_0_w },
	{ 0x8000, 0xffff, MWA_ROM },
MEMORY_END



/*************************************
 *
 *	Port definitions
 *
 *************************************/

#define PORT_SERVICE_NO_TOGGLE(mask,default)	\
	PORT_BITX(    mask, mask & default, IPT_SERVICE1, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )

#define UNUSED_ANALOG	\
	PORT_START	\
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )


INPUT_PORTS_START( stratab )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* input from sound board */
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Cabinet ))
	PORT_DIPSETTING(    0x08, DEF_STR( Upright ))
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ))
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* 60 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER2 | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START	/* 80 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* analog C */
    PORT_ANALOG( 0xff, 0x00, IPT_TRACKBALL_Y | IPF_PLAYER1 | IPF_CENTER, 25, 32, 0x80, 0x7f )

	PORT_START	/* analog D */
    PORT_ANALOG( 0xff, 0x00, IPT_TRACKBALL_X | IPF_PLAYER1 | IPF_REVERSE | IPF_CENTER, 25, 32, 0x80, 0x7f )

	PORT_START	/* analog E */
    PORT_ANALOG( 0xff, 0x00, IPT_TRACKBALL_Y | IPF_PLAYER2 | IPF_COCKTAIL | IPF_CENTER, 25, 32, 0x80, 0x7f )

	PORT_START	/* analog F */
    PORT_ANALOG( 0xff, 0x00, IPT_TRACKBALL_X | IPF_PLAYER2 | IPF_COCKTAIL | IPF_REVERSE | IPF_CENTER, 25, 32, 0x80, 0x7f )
INPUT_PORTS_END


INPUT_PORTS_START( wfortune )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* input from sound board */
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Cabinet ))
	PORT_DIPSETTING(    0x08, DEF_STR( Upright ))
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ))
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* 60 */
	PORT_BIT( 0x07, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START3 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER3 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN2 )

	PORT_START	/* 80 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	UNUSED_ANALOG	/* analog C */

	PORT_START		/* analog D */
	PORT_ANALOG( 0xff, 0x80, IPT_DIAL | IPF_PLAYER1, 75, 10, 0x00, 0xff )

	UNUSED_ANALOG	/* analog E */

	PORT_START		/* analog F */
	PORT_ANALOG( 0xff, 0x80, IPT_DIAL | IPF_PLAYER2 | IPF_COCKTAIL, 75, 10, 0x00, 0xff )
INPUT_PORTS_END


INPUT_PORTS_START( gtg )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* input from sound board */
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Cabinet ))
	PORT_DIPSETTING(    0x08, DEF_STR( Upright ))
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ))
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* 60 */
	/* it is still unknown how the second player inputs are muxed in */
	/* currently we map both sets of controls to the same inputs */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 )
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 | IPF_COCKTAIL )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 | IPF_COCKTAIL )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START	/* 80 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	UNUSED_ANALOG	/* analog C */
	UNUSED_ANALOG	/* analog D */
	UNUSED_ANALOG	/* analog E */
	UNUSED_ANALOG	/* analog F */
INPUT_PORTS_END


INPUT_PORTS_START( gtg2 )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_SERVICE_NO_TOGGLE( 0x02, IP_ACTIVE_LOW )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x30, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Cabinet ))
	PORT_DIPSETTING(    0x40, DEF_STR( Upright ))
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ))
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* 60 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT( 0x78, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START1 )

	PORT_START	/* 80 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER2 | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 | IPF_COCKTAIL )
	PORT_BIT( 0x78, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START2 )

	PORT_START	/* analog C */
    PORT_ANALOG( 0xff, 0x00, IPT_TRACKBALL_X | IPF_PLAYER1 | IPF_CENTER, 25, 32, 0x80, 0x7f )

	PORT_START	/* analog D */
    PORT_ANALOG( 0xff, 0x00, IPT_TRACKBALL_Y | IPF_PLAYER1 | IPF_REVERSE | IPF_CENTER, 25, 32, 0x80, 0x7f )

	PORT_START	/* analog E */
    PORT_ANALOG( 0xff, 0x00, IPT_TRACKBALL_X | IPF_PLAYER2 | IPF_COCKTAIL | IPF_CENTER, 25, 32, 0x80, 0x7f )

	PORT_START	/* analog F */
    PORT_ANALOG( 0xff, 0x00, IPT_TRACKBALL_Y | IPF_PLAYER2 | IPF_COCKTAIL | IPF_REVERSE | IPF_CENTER, 25, 32, 0x80, 0x7f )
INPUT_PORTS_END


INPUT_PORTS_START( gtg2t )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* input from sound board */
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Cabinet ))
	PORT_DIPSETTING(    0x08, DEF_STR( Upright ))
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ))
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* 60 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER2 | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START	/* 80 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* analog C */
    PORT_ANALOG( 0xff, 0x00, IPT_TRACKBALL_X | IPF_PLAYER1 | IPF_CENTER, 25, 32, 0x80, 0x7f )

	PORT_START	/* analog D */
    PORT_ANALOG( 0xff, 0x00, IPT_TRACKBALL_Y | IPF_PLAYER1 | IPF_REVERSE | IPF_CENTER, 25, 32, 0x80, 0x7f )

	PORT_START	/* analog E */
    PORT_ANALOG( 0xff, 0x00, IPT_TRACKBALL_X | IPF_PLAYER2 | IPF_COCKTAIL | IPF_CENTER, 25, 32, 0x80, 0x7f )

	PORT_START	/* analog F */
    PORT_ANALOG( 0xff, 0x00, IPT_TRACKBALL_Y | IPF_PLAYER2 | IPF_COCKTAIL | IPF_REVERSE | IPF_CENTER, 25, 32, 0x80, 0x7f )
INPUT_PORTS_END


INPUT_PORTS_START( slikshot )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* input from sound board */
	PORT_BIT( 0x7e, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* 60 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_BUTTON4       | IPF_PLAYER1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START	/* 80 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* analog C */
    PORT_ANALOG( 0xff, 0x00, IPT_TRACKBALL_X | IPF_PLAYER1 | IPF_CENTER, 25, 32, 0x80, 0x7f )

	PORT_START	/* analog D */
    PORT_ANALOG( 0xff, 0x00, IPT_TRACKBALL_Y | IPF_PLAYER1 | IPF_REVERSE | IPF_CENTER, 25, 32, 0x80, 0x7f )

	UNUSED_ANALOG	/* analog E */
	UNUSED_ANALOG	/* analog F */
INPUT_PORTS_END


INPUT_PORTS_START( arlingtn )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* input from sound board */
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ))	/* see code at e23c */
	PORT_DIPSETTING(    0x00, DEF_STR( Off ))
	PORT_DIPSETTING(    0x08, DEF_STR( On ))
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* 60 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START	/* 80 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON4        | IPF_PLAYER1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN2 )

	UNUSED_ANALOG	/* analog C */
	UNUSED_ANALOG	/* analog D */
	UNUSED_ANALOG	/* analog E */
	UNUSED_ANALOG	/* analog F */
INPUT_PORTS_END


INPUT_PORTS_START( peggle )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* input from sound board */
	PORT_BIT( 0x7e, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* 60 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 )
	PORT_BIT( 0x30, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START	/* 80 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	UNUSED_ANALOG	/* analog C */
	UNUSED_ANALOG	/* analog D */
	UNUSED_ANALOG	/* analog E */
	UNUSED_ANALOG	/* analog F */
INPUT_PORTS_END


INPUT_PORTS_START( pegglet )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* input from sound board */
	PORT_BIT( 0x7e, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* 60 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x3e, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START	/* 80 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	UNUSED_ANALOG	/* analog C */

	PORT_START		/* analog D */
	PORT_ANALOG( 0xff, 0x00, IPT_DIAL | IPF_PLAYER1 | IPF_CENTER, 50, 10, 0x80, 0x7f )

	UNUSED_ANALOG	/* analog E */
	UNUSED_ANALOG	/* analog F */
INPUT_PORTS_END


INPUT_PORTS_START( hstennis )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* input from sound board */
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Cabinet ))
	PORT_DIPSETTING(    0x08, DEF_STR( Upright ))
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ))
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ))	/* see code at fbb5 */
	PORT_DIPSETTING(    0x00, DEF_STR( Off ))
	PORT_DIPSETTING(    0x10, DEF_STR( On ))
	PORT_BIT( 0x60, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* 60 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START	/* 80 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN2 )

	UNUSED_ANALOG	/* analog C */
	UNUSED_ANALOG	/* analog D */
	UNUSED_ANALOG	/* analog E */
	UNUSED_ANALOG	/* analog F */
INPUT_PORTS_END


INPUT_PORTS_START( rimrockn )
	PORT_START	/* 40 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* input from sound board */
	PORT_BIT( 0xfe, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* 60 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_SERVICE_NO_TOGGLE( 0x02, IP_ACTIVE_LOW )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* 80 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* special 161 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START1 )

	PORT_START	/* special 162 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START2 )

	PORT_START	/* special 163 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER3 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER3 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER3 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER3 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER3 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START3 )

	PORT_START	/* special 164 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER4 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER4 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER4 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER4 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER4 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER4 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START4 )

	PORT_START	/* special 165 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


INPUT_PORTS_START( ninclown )
	PORT_START	/* 40 */
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_SERVICE_NO_TOGGLE( 0x0200, IP_ACTIVE_LOW )
	PORT_BIT( 0x0c00, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0xc000, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* 60 */
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER1 )
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 )
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 )
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 )
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )

	PORT_START	/* 80 */
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER2 )
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 )
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 )
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 )
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER2 )
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 )

	UNUSED_ANALOG	/* analog C */
	UNUSED_ANALOG	/* analog D */
	UNUSED_ANALOG	/* analog E */
	UNUSED_ANALOG	/* analog F */
INPUT_PORTS_END



/*************************************
 *
 *	Sound definitions
 *
 *************************************/

static struct YM2203interface ym2203_interface =
{
	1,
	CLOCK_8MHz/2,
	{ YM2203_VOL(75,25) },
	{ 0 },
	{ 0 },
	{ 0 },
	{ ym2203_portb_out },
	{ generate_sound_irq }
};


static struct YM3812interface ym3812_interface =
{
	1,
	CLOCK_8MHz/2,
	{ 75 },
	{ generate_sound_irq }
};


static struct OKIM6295interface oki6295_interface_low =
{
	1,
	{ CLOCK_8MHz/8/165 },
	{ REGION_SOUND1 },
	{ 75 }
};


static struct OKIM6295interface oki6295_interface_high =
{
	1,
	{ CLOCK_8MHz/8/128 },
	{ REGION_SOUND1 },
	{ 75 }
};



/*************************************
 *
 *	Machine driver
 *
 *************************************/

#define ITECH_DRIVER(NAME, CPUTYPE, CPUCLOCK, MAINMEM, YMTYPE, OKISPEED, XMIN, XMAX)	\
static struct MachineDriver machine_driver_##NAME =								\
{																				\
	/* basic machine hardware */												\
	{																			\
		{																		\
			CPU_##CPUTYPE,														\
			CPUCLOCK,															\
			MAINMEM##_readmem,MAINMEM##_writemem,0,0,							\
			generate_nmi,1														\
		},																		\
		{																		\
			CPU_M6809,															\
			CLOCK_8MHz/4,														\
			sound##YMTYPE##_readmem,sound##YMTYPE##_writemem,0,0,				\
			ignore_interrupt,1													\
		}																		\
	},																			\
	60,(int)(((263. - 240.) / 263.) * 1000000. / 60.),							\
	1,																			\
	init_machine,																\
																				\
	/* video hardware */														\
	512, 263, { XMIN, XMAX, 0, 239 },											\
	0,																			\
	256,256,																	\
	0,																			\
																				\
	VIDEO_TYPE_RASTER | VIDEO_MODIFIES_PALETTE | VIDEO_UPDATE_BEFORE_VBLANK,	\
	0,																			\
	itech8_vh_start,															\
	itech8_vh_stop,																\
	itech8_vh_screenrefresh,													\
																				\
	/* sound hardware */														\
	0,0,0,0,																	\
	{																			\
		{ SOUND_YM##YMTYPE, &ym##YMTYPE##_interface },							\
		{ SOUND_OKIM6295, &oki6295_interface_##OKISPEED },						\
	},																			\
	nvram_handler																\
}


/*           NAME,      CPU,    CPUCLOCK,      MAINMEM,  YMTYPE, OKISPEED, XMIN, XMAX) */
ITECH_DRIVER(tmslo2203, M6809,  CLOCK_8MHz/4,  tmslo,    2203,   high,     0,    255);
ITECH_DRIVER(tmshi2203, M6809,  CLOCK_8MHz/4,  tmshi,    2203,   high,     0,    255);
ITECH_DRIVER(gtg2,      M6809,  CLOCK_8MHz/4,  gtg2,     3812,   high,     0,    255);
ITECH_DRIVER(peggle,    M6809,  CLOCK_8MHz/4,  tmslo,    3812,   high,     18,   367);
ITECH_DRIVER(arlingtn,  M6809,  CLOCK_8MHz/4,  tmshi,    3812,   low,      16,   389);
ITECH_DRIVER(hstennis,  M6809,  CLOCK_8MHz/4,  tmshi,    3812,   high,     0,    375);
ITECH_DRIVER(rimrockn,  M6809,  CLOCK_12MHz/4, tmshi,    3812,   high,     16,   375);
ITECH_DRIVER(ninclown,  M68000, CLOCK_12MHz,   ninclown, 3812,   high,     64,   423);



/*************************************
 *
 *	ROM definitions
 *
 *************************************/

ROM_START( stratab )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "sbprogv3.bin", 0x08000, 0x8000, 0xa5ae728f )
	ROM_COPY( REGION_CPU1,    0x8000, 0x14000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "sbsnds.bin", 0x08000, 0x8000, 0xb36c8f0a )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, 0xa915b0bd )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, 0x340c661f )
	ROM_LOAD( "grom2.bin", 0x40000, 0x20000, 0x5df9f1cf )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0.bin", 0x00000, 0x20000, 0x6ff390b9 )
ROM_END


ROM_START( wfortune )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "wofpgm", 0x04000, 0x4000, 0xbd984654 )
	ROM_CONTINUE(       0x10000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x14000, 0x8000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "wofsnd", 0x08000, 0x8000, 0x0a6aa5dc )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "wofgrom0", 0x00000, 0x10000, 0x9a157b2c )
	ROM_LOAD( "wofgrom1", 0x10000, 0x10000, 0x5064739b )
	ROM_LOAD( "wofgrom2", 0x20000, 0x10000, 0x3d393b2b )
	ROM_LOAD( "wofgrom3", 0x30000, 0x10000, 0x117a2ce9 )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "wofsbom0", 0x00000, 0x20000, 0x5c28c3fe )
ROM_END


ROM_START( gtg )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "u5.bin", 0x04000, 0x4000, 0x61984272 )
	ROM_CONTINUE(       0x10000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x14000, 0x8000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u27.bin", 0x08000, 0x8000, 0x358d2440 )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, 0xa29c688a )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, 0xb52a23f6 )
	ROM_LOAD( "grom2.bin", 0x40000, 0x20000, 0x9b8e3a61 )
	ROM_LOAD( "grom3.bin", 0x60000, 0x20000, 0xb6e9fb15 )
	ROM_LOAD( "grom4.bin", 0x80000, 0x20000, 0xfaa16729 )
	ROM_LOAD( "grom5.bin", 0xa0000, 0x20000, 0x5b393314 )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0.bin", 0x00000, 0x20000, 0x1cccbfdf )
ROM_END


ROM_START( slikshot )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "u5.bin", 0x04000, 0x4000, 0x00000000 )
	ROM_CONTINUE(       0x10000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x14000, 0x8000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u27.bin", 0x08000, 0x8000, 0xa96ce0f7 )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, 0xe60c2804 )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, 0xd764d542 )

	ROM_REGION( 0x10000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0.bin", 0x00000, 0x10000, 0x4b075f5e )
ROM_END


ROM_START( gtg2 )
	/* banks are loaded in the opposite order from the others, */
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "u5.2",   0x10000, 0x4000, 0x4a61580f )
	ROM_CONTINUE(       0x04000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x8000, 0x14000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u27.2", 0x08000, 0x8000, 0x55734876 )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, 0xa29c688a )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, 0xa4182776 )
	ROM_LOAD( "grom2.bin", 0x40000, 0x20000, 0x0580bb99 )
	ROM_LOAD( "grom3.bin", 0x60000, 0x20000, 0x89edb624 )
	ROM_LOAD( "grom4.bin", 0x80000, 0x20000, 0xf6557950 )
	ROM_LOAD( "grom5.bin", 0xa0000, 0x20000, 0xa680ce6a )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "vr-srom0", 0x00000, 0x20000, 0x4dd4db42 )
ROM_END


ROM_START( gtg2t )
	/* banks are loaded in the opposite order from the others, */
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "u5",     0x10000, 0x4000, 0xc7b3a9f3 )
	ROM_CONTINUE(       0x04000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x8000, 0x14000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u27.bin", 0x08000, 0x8000, 0xdd2a5905 )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, 0xa29c688a )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, 0xa4182776 )
	ROM_LOAD( "grom2.bin", 0x40000, 0x20000, 0x0580bb99 )
	ROM_LOAD( "grom3.bin", 0x60000, 0x20000, 0x89edb624 )
	ROM_LOAD( "grom4.bin", 0x80000, 0x20000, 0xf6557950 )
	ROM_LOAD( "grom5.bin", 0xa0000, 0x20000, 0xa680ce6a )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "vr-srom0", 0x00000, 0x20000, 0x4dd4db42 )
ROM_END


ROM_START( gtg2j )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "u5.bin", 0x04000, 0x4000, 0x9c95ceaa )
	ROM_CONTINUE(       0x10000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x14000, 0x8000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u27.bin", 0x08000, 0x8000, 0xdd2a5905 )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, 0xa29c688a )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, 0xa4182776 )
	ROM_LOAD( "grom2.bin", 0x40000, 0x20000, 0x0580bb99 )
	ROM_LOAD( "grom3.bin", 0x60000, 0x20000, 0x89edb624 )
	ROM_LOAD( "grom4.bin", 0x80000, 0x20000, 0xf6557950 )
	ROM_LOAD( "grom5.bin", 0xa0000, 0x20000, 0xa680ce6a )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0.bin", 0x00000, 0x20000, 0x1cccbfdf )
ROM_END


ROM_START( arlingtn )
	/* banks are loaded in the opposite order from the others, */
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "ahrd121.bin", 0x10000, 0x4000, 0x00aae02e )
	ROM_CONTINUE(            0x04000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x8000, 0x14000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "ahrsnd11.bin", 0x08000, 0x8000, 0xdec57dca )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, 0x5ef57fe5 )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, 0x6aca95c0 )
	ROM_LOAD( "grom2.bin", 0x40000, 0x10000, 0x6d6fde1b )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0.bin", 0x00000, 0x40000, 0x56087f81 )
ROM_END


ROM_START( peggle )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "j-stick.u5", 0x04000, 0x4000, 0x140d5a9c )
	ROM_CONTINUE(           0x10000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x14000, 0x8000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "sound.u27", 0x08000, 0x8000, 0xb99beb70 )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, 0x5c02348d )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, 0x85a7a3a2 )
	ROM_LOAD( "grom2.bin", 0x40000, 0x20000, 0xbfe11f18 )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0", 0x00000, 0x20000, 0x001846ea )
ROM_END


ROM_START( pegglet )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "trakball.u5", 0x04000, 0x4000, 0xd2694868 )
	ROM_CONTINUE(            0x10000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x14000, 0x8000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "sound.u27", 0x08000, 0x8000, 0xb99beb70 )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, 0x5c02348d )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, 0x85a7a3a2 )
	ROM_LOAD( "grom2.bin", 0x40000, 0x20000, 0xbfe11f18 )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0", 0x00000, 0x20000, 0x001846ea )
ROM_END


ROM_START( hstennis )
	ROM_REGION( 0x1c000, REGION_CPU1, 0 )
	ROM_LOAD( "ten_v1_1.bin", 0x04000, 0x4000, 0xfaffab5c )
	ROM_CONTINUE(             0x10000, 0xc000 )
	ROM_COPY( REGION_CPU1, 0x14000, 0x8000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "tensd_v1.bin", 0x08000, 0x8000, 0xf034a694 )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, 0x1e69ebae )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, 0x4e6a22d5 )
	ROM_LOAD( "grom2.bin", 0x40000, 0x20000, 0xc0b643a9 )
	ROM_LOAD( "grom3.bin", 0x60000, 0x20000, 0x54afb456 )
	ROM_LOAD( "grom4.bin", 0x80000, 0x20000, 0xee09d645 )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0.bin", 0x00000, 0x20000, 0xd9ce58c3 )
ROM_END


ROM_START( rimrockn )
	ROM_REGION( 0x34000, REGION_CPU1, 0 )
	ROM_LOAD( "u5-2_2",    0x04000, 0x4000, 0x97777683 )
	ROM_CONTINUE(          0x10000, 0x4000 )
	ROM_CONTINUE(          0x1c000, 0x4000 )
	ROM_CONTINUE(          0x28000, 0xc000 )
	ROM_CONTINUE(          0x2c000, 0x8000 )
	ROM_COPY( REGION_CPU1, 0x2c000, 0x08000, 0x8000 )
	ROM_COPY( REGION_CPU1, 0x2c000, 0x14000, 0x8000 )
	ROM_COPY( REGION_CPU1, 0x2c000, 0x20000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u27", 0x08000, 0x8000, 0x59f87f0e )

	ROM_REGION( 0x100000, REGION_GFX1, 0 )
	ROM_LOAD( "grom00",       0x00000, 0x40000, 0x3eacbad9 )
	ROM_LOAD( "grom01",       0x40000, 0x40000, 0x864cc269 )
	ROM_LOAD( "grom02-2.st2", 0x80000, 0x40000, 0x47904233 )
	ROM_LOAD( "grom03-2.st2", 0xc0000, 0x40000, 0xf005f118 )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0", 0x00000, 0x40000, 0x7ad42be0 )
ROM_END


ROM_START( rimrck20 )
	ROM_REGION( 0x34000, REGION_CPU1, 0 )
	ROM_LOAD( "rrb.bin",   0x04000, 0x4000, 0x7e9d5545 )
	ROM_CONTINUE(          0x10000, 0x4000 )
	ROM_CONTINUE(          0x1c000, 0x4000 )
	ROM_CONTINUE(          0x28000, 0xc000 )
	ROM_CONTINUE(          0x2c000, 0x8000 )
	ROM_COPY( REGION_CPU1, 0x2c000, 0x08000, 0x8000 )
	ROM_COPY( REGION_CPU1, 0x2c000, 0x14000, 0x8000 )
	ROM_COPY( REGION_CPU1, 0x2c000, 0x20000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u27", 0x08000, 0x8000, 0x59f87f0e )

	ROM_REGION( 0x100000, REGION_GFX1, 0 )
	ROM_LOAD( "grom00",       0x00000, 0x40000, 0x3eacbad9 )
	ROM_LOAD( "grom01",       0x40000, 0x40000, 0x864cc269 )
	ROM_LOAD( "grom02-2.st2", 0x80000, 0x40000, 0x47904233 )
	ROM_LOAD( "grom03-2.st2", 0xc0000, 0x40000, 0xf005f118 )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0", 0x00000, 0x40000, 0x7ad42be0 )
ROM_END


ROM_START( rimrck12 )
	ROM_REGION( 0x34000, REGION_CPU1, 0 )
	ROM_LOAD( "rrbbv12.u5",0x04000, 0x4000, 0x661761a6 )
	ROM_CONTINUE(          0x10000, 0x4000 )
	ROM_CONTINUE(          0x1c000, 0x4000 )
	ROM_CONTINUE(          0x28000, 0xc000 )
	ROM_CONTINUE(          0x2c000, 0x8000 )
	ROM_COPY( REGION_CPU1, 0x2c000, 0x08000, 0x8000 )
	ROM_COPY( REGION_CPU1, 0x2c000, 0x14000, 0x8000 )
	ROM_COPY( REGION_CPU1, 0x2c000, 0x20000, 0x8000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "rrbsndv1.u27", 0x08000, 0x8000, 0x8eda5f53 )

	ROM_REGION( 0x100000, REGION_GFX1, 0 )
	ROM_LOAD( "grom00", 0x00000, 0x40000, 0x3eacbad9 )
	ROM_LOAD( "grom01", 0x40000, 0x40000, 0x864cc269 )
	ROM_LOAD( "grom02", 0x80000, 0x40000, 0x34e567d5 )
	ROM_LOAD( "grom03", 0xc0000, 0x40000, 0xfd18045d )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )
	ROM_LOAD( "srom0", 0x00000, 0x40000, 0x7ad42be0 )
ROM_END


ROM_START( ninclown )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )
	ROM_LOAD16_BYTE( "prog1", 0x00000, 0x20000, 0xfabfdcd2 )
	ROM_LOAD16_BYTE( "prog0", 0x00001, 0x20000, 0xeca63db5 )
	ROM_COPY(    REGION_CPU1, 0x08000, 0x40000, 0x38000 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "nc-snd", 0x08000, 0x8000, 0xf9d5b4e1 )

	ROM_REGION( 0x180000, REGION_GFX1, 0 )
	ROM_LOAD( "nc-grom0", 0x000000, 0x40000, 0x532f7bff )
	ROM_LOAD( "nc-grom1", 0x040000, 0x40000, 0x45640d4a )
	ROM_LOAD( "nc-grom2", 0x080000, 0x40000, 0xc8281d06 )
	ROM_LOAD( "nc-grom3", 0x0c0000, 0x40000, 0x2a6d33ac )
	ROM_LOAD( "nc-grom4", 0x100000, 0x40000, 0x910876ba )
	ROM_LOAD( "nc-grom5", 0x140000, 0x40000, 0x2533279b )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )
	ROM_LOAD( "nc-srom0", 0x00000, 0x40000, 0x00000000 )
ROM_END



/*************************************
 *
 *	Driver-specific init
 *
 *************************************/

static void init_viasound(void)
{
	/* some games with a YM3812 use a VIA(6522) for timing and communication */
	install_mem_read_handler (1, 0x5000, 0x500f, via6522_r);
	via6522 = install_mem_write_handler(1, 0x5000, 0x500f, via6522_w);
}


static void init_rimrockn(void)
{
	/* additional input ports */
	install_mem_read_handler (0, 0x0161, 0x0161, input_port_3_r);
	install_mem_read_handler (0, 0x0162, 0x0162, input_port_4_r);
	install_mem_read_handler (0, 0x0163, 0x0163, input_port_5_r);
	install_mem_read_handler (0, 0x0164, 0x0164, input_port_6_r);
	install_mem_read_handler (0, 0x0165, 0x0165, input_port_7_r);

	/* different banking mechanism (disable the old one) */
	install_mem_write_handler(0, 0x01a0, 0x01a0, rimrockn_bank_w);
	install_mem_write_handler(0, 0x01c0, 0x01df, itech8_blitter_w);

	/* VIA-based sound timing */
	init_viasound();
}



/*************************************
 *
 *	Game drivers
 *
 *************************************/

GAME ( 1989, wfortune, 0,        tmshi2203, wfortune, 0,        ROT0,   "GameTek", "Wheel Of Fortune" )
GAME ( 1990, stratab,  0,        tmshi2203, stratab,  0,        ROT270, "Strata/Incredible Technologies", "Strata Bowling" )
GAME ( 1990, gtg,      0,        tmshi2203, gtg,      0,        ROT0,   "Strata/Incredible Technologies", "Golden Tee Golf" )
GAMEX( 1990, slikshot, 0,        tmshi2203, slikshot, 0,        ROT90,  "Grand Products/Incredible Technologies", "Slick Shot", GAME_NOT_WORKING )
GAME ( 1990, hstennis, 0,        hstennis,  hstennis, 0,        ROT90,  "Strata/Incredible Technologies", "Hot Shots Tennis" )
GAME ( 1991, arlingtn, 0,        arlingtn,  arlingtn, 0,        ROT0,   "Strata/Incredible Technologies", "Arlington Horse Racing" )
GAME ( 1991, peggle,   0,        peggle,    peggle,   0,        ROT90,  "Strata/Incredible Technologies", "Peggle (Joystick)" )
GAME ( 1991, pegglet,  peggle,   peggle,    pegglet,  0,        ROT90,  "Strata/Incredible Technologies", "Peggle (Trackball)" )
GAME ( 1991, rimrockn, 0,        rimrockn,  rimrockn, rimrockn, ROT0,   "Strata/Incredible Technologies", "Rim Rockin' Basketball (V2.2)" )
GAME ( 1991, rimrck20, rimrockn, rimrockn,  rimrockn, rimrockn, ROT0,   "Strata/Incredible Technologies", "Rim Rockin' Basketball (V2.0)" )
GAME ( 1991, rimrck12, rimrockn, rimrockn,  rimrockn, rimrockn, ROT0,   "Strata/Incredible Technologies", "Rim Rockin' Basketball (V1.2)" )
GAME ( 1991, ninclown, 0,        ninclown,  ninclown, viasound, ROT0,   "Strata/Incredible Technologies", "Ninja Clowns" )
GAME ( 1992, gtg2,     0,        gtg2,      gtg2,     viasound, ROT0,   "Strata/Incredible Technologies", "Golden Tee Golf II (Trackball, V2.2)" )
GAME ( 1989, gtg2t,    gtg2,     tmshi2203, gtg2t,    0,        ROT0,   "Strata/Incredible Technologies", "Golden Tee Golf II (Trackball, V1.1)" )
GAME ( 1991, gtg2j,    gtg2,     tmslo2203, gtg,      0,        ROT0,   "Strata/Incredible Technologies", "Golden Tee Golf II (Joystick, V1.0)" )
