#include "osdepend.h"
#include "unzip.h"
#include "zlib.h"
#include "fileio.h"
#include "options.h"

#include <stdio.h>
#include <malloc.h>
#include <io.h>
#include <unistd.h>
#include <string.h>

#define logerror myerrorlog

#define LOG_CORRUPT_POINTER( func ) {                      \
  logerror( "Corrupt file pointer passed to %s.", func );  \
}

#define CHECK_FILE_NULL( func, badrc ) {                                    \
  if ( !file )                                                              \
  {                                                                         \
    logerror( "Attempted to access a NULL file handle with %s.", func );    \
    return badrc;                                                           \
  }                                                                         \
}

extern void CLIB_DECL logerror( const char *text, ... );

void * osd_fopen( const char *gamename, const char *filename, int filetype, int rdwr )
{
  struct MAMEFILE *mf = NULL;
  struct MAMEpaths *thepath;
  FILE *tempfile = NULL;
  char found, usefn, *suffix;
  unsigned char *tmpdata;
  char buffer[ 1024 ], buffer2[ 1024 ];
  unsigned int i, tmplen;

//  if ( gamename && filename ) logerror( "Open %s / %s %d - %c.", gamename, filename, filetype, (rdwr ? 'W' : 'R') );
//  else if ( filename ) logerror( "Open %s %d - %c.", filename, filetype, (rdwr ? 'W' : 'R') );
//  else logerror( "Open %s %d - %c.", gamename, filetype, (rdwr ? 'W' : 'R') );

  // usefn values:
  // 0 = use gamename and not filename
  // 1 = use filename and gamename
  // 2 = can use filename only
  // 3 = must use filename only

  switch( filetype )
  {
    case OSD_FILETYPE_ROM:
      if ( rdwr )
      {
        logerror( "Attempted to write to a sample file!: %s",
          filename );
        return NULL;
      }
      thepath = globaloptions.rompath;
      usefn = 1;
      suffix = "\0";
    break;
    case OSD_FILETYPE_SAMPLE:
      if ( rdwr )
      {
        logerror( "Attempted to write to a ROM or sample!: %s",
          filename );
        return NULL;
      }
      thepath = globaloptions.samplepath;
      usefn = 1;
      suffix = "\0";
    break;

    case OSD_FILETYPE_NVRAM:
      thepath = globaloptions.statepath;
      usefn = 0;
      suffix = ".nv";
    break;

    case OSD_FILETYPE_HIGHSCORE:
      thepath = globaloptions.statepath;
      usefn = 0;
      suffix = ".hi";
    break;

    case OSD_FILETYPE_CONFIG:
      thepath = globaloptions.statepath;
      usefn = 0;
      suffix = ".cfg";
    break;

    case OSD_FILETYPE_INPUTLOG:
      thepath = globaloptions.statepath;
      usefn = 0;
      suffix = ".inp";
    break;

    case OSD_FILETYPE_STATE:
      thepath = globaloptions.statepath;
      usefn = 0;
      suffix = ".sta";
    break;

    case OSD_FILETYPE_ARTWORK:
      if ( rdwr )
      {
        logerror( "Attempted to write to an artwork file!: %s",
          filename );
        return NULL;
      }
      thepath = globaloptions.artpath;
      usefn = 2;
      suffix = "\0";
    break;

    case OSD_FILETYPE_MEMCARD:
      thepath = globaloptions.statepath;
      usefn = 3;
      suffix = "\0";
    break;

    case OSD_FILETYPE_SCREENSHOT:
      if ( !rdwr )
      {
        logerror( "Attempted to read a screenshot file for some goofy reason." );
        return NULL;
      }
      thepath = globaloptions.capturepath;
      usefn = 3;
      suffix = ".png";
    break;

    case OSD_FILETYPE_HIGHSCORE_DB:
    case OSD_FILETYPE_HISTORY:
    case OSD_FILETYPE_CHEAT:
      if ( rdwr )
      {
        logerror( "Attempted to write to a HiScore DB, History, or Cheat file: %s",
          filename );
        return NULL;
      }
      thepath = globaloptions.curpath;
      usefn = 3;
      suffix = "\0";
    break;

    case OSD_FILETYPE_LANGUAGE:
      if ( rdwr )
      {
        logerror( "Attempted to write to a language file: %s",
          filename );
        return NULL;
      }
      thepath = globaloptions.curpath;
      usefn = 3;
      suffix = ".lng";
    break;

    default:
      logerror( "Attempted to open unknown file type: %d",
        filetype );
      return NULL;
  }

  found = 0;

  for ( i = 0; i < thepath->numpaths && !found; ++i )
  {
    switch ( usefn )
    {
      case 0:
        // Uses gamename only.
        // Look for path\gamename.suffix
        // Don't look for a ZIP file

        sprintf( buffer, "%s\\%s%s", thepath->paths[i], 
         gamename, suffix );

        tempfile = fopen( buffer, (rdwr ? "wb" : "rb") );
        if ( tempfile )
        {
          found = 1;
          break;
        }
      break;
      case 1:
        // Use gamename and filename.
        // Look for path\gamename\filename.suffix
        // Look for path\gamename.zip -> filename.suffix

        sprintf( buffer, "%s\\%s\\%s%s", thepath->paths[i], 
         gamename, filename, suffix );

        tempfile = fopen( buffer, (rdwr ? "wb" : "rb") );
        if ( tempfile )
        {
          found = 1;
          break;
        }

        if ( rdwr ) continue; // Zip files are read-only to MAME

        sprintf( buffer, "%s\\%s.zip", thepath->paths[i], 
         gamename );

        sprintf( buffer2, "%s%s", filename, suffix );

        if ( access( buffer, R_OK ) == 0 )
        {
          if ( load_zipped_file( buffer, buffer2, &tmpdata, 
            &tmplen ) == 0 )
          {
            found = 2;
            break;
          }
        }
      break;
      case 2:
        // Use gamename and filename or filename only
        // Look for path\filename.suffix
        // Look for path\gamename\filename.suffix
        // Look for path\filename.zip -> filename.suffix
        // Look for path\gamename.zip -> filename.suffix
        // Look for path\gamename\filename.zip -> filename.suffix

        sprintf( buffer, "%s\\%s%s", thepath->paths[i],
         filename, suffix );

        tempfile = fopen( buffer, (rdwr ? "wb" : "rb") );
        if ( tempfile )
        {
          found = 1;
          break;
        }

        sprintf( buffer, "%s\\%s\\%s%s", thepath->paths[i],
         gamename, filename, suffix );

        tempfile = fopen( buffer, (rdwr ? "wb" : "rb") );
        if ( tempfile )
        {
          found = 1;
          break;
        }

        if ( rdwr ) continue; // Zip files are read-only to MAME

        sprintf( buffer, "%s\\%s.zip", thepath->paths[i],
         filename );

        sprintf( buffer2, "%s%s", filename, suffix );

        if ( access( buffer, R_OK ) == 0 )
        {
          if ( load_zipped_file( buffer, buffer2, &tmpdata, 
            &tmplen ) == 0 )
          {
            found = 2;
            break;
          }
        }
        sprintf( buffer, "%s\\%s.zip", thepath->paths[i],
         gamename );

        if ( access( buffer, R_OK ) == 0 )
        {
          if ( load_zipped_file( buffer, buffer2, &tmpdata, 
            &tmplen ) == 0 )
          {
            found = 2;
            break;
          }
        }

        sprintf( buffer, "%s\\%s\\%s.zip", thepath->paths[i],
         gamename, filename );

        if ( access( buffer, R_OK ) == 0 )
        {
          if ( load_zipped_file( buffer, buffer2, &tmpdata, 
            &tmplen ) == 0 )
          {
            found = 2;
            break;
          }
        }
      break;
      case 3:
        // Use filename without gamename
        // Look for path\filename.suffix
        // Look for path\filename.zip -> filename.suffix

        sprintf( buffer, "%s\\%s%s", thepath->paths[i],
         filename, suffix );

        tempfile = fopen( buffer, (rdwr ? "wb" : "rb") );
        if ( tempfile )
        {
          found = 1;
          break;
        }

        if ( rdwr ) continue; // Zip files are read-only to MAME

        sprintf( buffer, "%s\\%s.zip", thepath->paths[i],
         filename );

        sprintf( buffer2, "%s%s", filename, suffix );

        if ( access( buffer, R_OK ) == 0 )
        {
          if ( load_zipped_file( buffer, buffer2, &tmpdata, 
            &tmplen ) == 0 )
          {
            found = 2;
            break;
          }
        }
      break;
    }
  }

  switch ( found )
  {
    case 1:
      mf = (struct MAMEFILE *)malloc( sizeof( struct MAMEFILE ) );
      if ( !mf ) return NULL;
      mf->fp = tempfile;
      mf->filetype = 0;
      mf->extradata = NULL;
      mf->filelength = filelength( fileno( tempfile ) );
      mf->fileposition = 0;
      mf->crc_calcd = 0;
//      logerror( "Found." );
    break;
    case 2:
      mf = (struct MAMEFILE *)malloc( sizeof( struct MAMEFILE ) );
      if ( !mf ) return NULL;
      mf->fp = NULL;
      mf->filetype = 1;
      mf->extradata = tmpdata;
      mf->filelength = tmplen;
      mf->fileposition = 0;
      mf->crc_calcd = 0;
//      logerror( "Found in ZIP file." );
    break;
    default:
      mf = NULL;
//      logerror( "Not found." );
  }

  return mf;
}

int osd_fseek( void *file, int offset, int whence )
{
  int rc = 0;
  struct MAMEFILE *mf = (struct MAMEFILE *)file;

  CHECK_FILE_NULL( "osd_fseek", -1 );

  switch ( mf->filetype )
  {
    case 0:
      rc = fseek( mf->fp, offset, whence );
    break;
    case 1:
      switch ( whence )
      {
        case SEEK_SET:
          if ( offset > mf->filelength )
          {
            rc = -1;
          } else {
            mf->fileposition = offset;
          }
        break;
        case SEEK_CUR:
          if ( mf->fileposition + offset > mf->filelength )
          {
            rc = -1;
          } else {
            mf->fileposition += offset;
          }
        break;
        case SEEK_END:
          if ( mf->filelength - offset < 0 )
          {
            rc = -1;
          } else {
            mf->fileposition = mf->filelength - offset;
          }
        break;
        default:
          logerror( "Bad osd_fseek operation was requested." );
          return -1;
      }
    break;
    default:
      LOG_CORRUPT_POINTER( "osd_fseek" );
      return -1;
  }

  return rc;
}

int osd_ftell( void *file )
{
  struct MAMEFILE *mf = (struct MAMEFILE *)file;

  CHECK_FILE_NULL( "osd_ftell", -1 );

  switch ( mf->filetype )
  {
    case 0:
      return ftell( mf->fp );
    case 1:
      return mf->fileposition;
    default:
      LOG_CORRUPT_POINTER( "osd_ftell" );
      return -1;
  }
}

int osd_fread( void *file, void *buffer, int length )
{
  struct MAMEFILE *mf = (struct MAMEFILE *)file;
  int reallength = length;

  CHECK_FILE_NULL( "osd_fread", 0 );

  switch ( mf->filetype )
  {
    case 0:
      return fread( buffer, 1, length, mf->fp );
    case 1:
      if ( mf->fileposition + length > mf->filelength )
      {
        reallength = mf->filelength - mf->fileposition;
      }
      memcpy( buffer, mf->extradata, reallength );
      mf->fileposition += reallength;
      return reallength;
    default:
      LOG_CORRUPT_POINTER( "osd_fread" );
      return 0;
  }

}

int osd_fread_swap( void *file, void *buffer, int length )
{
  int i;
  unsigned char *buf;
  unsigned char temp;
  int res;

  res = osd_fread (file, buffer, length);

  buf = buffer;
  for( i = 0; i < length; i += 2 )
  {
    temp = buf[i];
    buf[i] = buf[i + 1];
    buf[i + 1] = temp;
  }

  return res;
}

int osd_fwrite( void *file, const void *buffer, int length )
{
  struct MAMEFILE *mf = (struct MAMEFILE *)file;

  CHECK_FILE_NULL( "osd_fwrite", 0 );

  switch ( mf->filetype )
  {
    case 0:
      return fwrite( buffer, 1, length, mf->fp );
    case 1:
      return 0;  // Write not supported for ZIP files
    default:
      LOG_CORRUPT_POINTER( "osd_fwrite" );
      return 0;
  }
}

int osd_fwrite_swap( void *file, const void *buffer, int length )
{
  int i;
  unsigned char *buf;
  unsigned char temp;
  int res;

  buf = (unsigned char *) buffer;
  for( i = 0; i < length; i += 2 )
  {
    temp = buf[i];
    buf[i] = buf[i + 1];
    buf[i + 1] = temp;
  }

  res = osd_fwrite (file, buffer, length);

  for( i = 0; i < length; i += 2 )
  {
    temp = buf[i];
    buf[i] = buf[i + 1];
    buf[i + 1] = temp;
  }

  return res;
}

int osd_fread_scatter( void *file, void *buffer, int length, int increment )
{
  unsigned char *buf = buffer;
  struct MAMEFILE *mf = (struct MAMEFILE *)file;
  unsigned char tempbuf[4096];
  int totread, r, i;

  CHECK_FILE_NULL( "osd_fread_scatter", 0 );

  switch( mf->filetype )
  {
    case 0:
      totread = 0;
      while (length)
      {
        r = length;
        if( r > 4096 )
          r = 4096;
        r = fread (tempbuf, 1, r, mf->fp);
        if( r == 0 )
          return totread;       /* error */
        for( i = 0; i < r; i++ )
        {
          *buf = tempbuf[i];
          buf += increment;
        }
        totread += r;
        length -= r;
      }
      return totread;
    break;
    case 1:
      if( mf->extradata )
      {
        if( length + mf->fileposition > mf->filelength )
          length = mf->filelength - mf->fileposition;
        for( i = 0; i < length; i++ )
        {
          *buf = mf->extradata[mf->fileposition + i];
          buf += increment;
        }
        mf->fileposition += length;
        return length;
      }
    break;
    default:
      LOG_CORRUPT_POINTER( "osd_fread_scatter" );
  }

  return 0;
}

int osd_fsize( void *file )
{
  struct MAMEFILE *mf = (struct MAMEFILE *)file;

  CHECK_FILE_NULL( "osd_fsize", 0 );

  return mf->filelength;
}

unsigned int osd_fcrc( void *file )
{
  struct MAMEFILE *mf = (struct MAMEFILE *)file;
  unsigned char *tmpdata;
  unsigned long curpos;

  CHECK_FILE_NULL( "osd_fcrc", 0 );

  if ( !mf->crc_calcd ) 
  {
    switch ( mf->filetype )
    {
      case 0:
        curpos = ftell( mf->fp );
        fseek( mf->fp, 0, SEEK_SET );
        tmpdata = (unsigned char *)malloc( mf->filelength );
        fread( tmpdata, 1, mf->filelength, mf->fp );
        mf->crc = crc32( 0, tmpdata, mf->filelength );
        free( tmpdata );
        fseek( mf->fp, curpos, SEEK_SET );
      break;
      case 1:
        mf->crc = crc32( 0, mf->extradata, mf->filelength );
      break;
      default:
        LOG_CORRUPT_POINTER( "osd_fcrc" );
        return 0;
    }

    mf->crc_calcd = 1;
  }

  return mf->crc;
}

int osd_fchecksum( const char *gamename, const char *filename, unsigned int *length,
  unsigned int *sum )
{
  struct MAMEFILE *mf;

  mf = osd_fopen( gamename, filename, OSD_FILETYPE_ROM, 0 );
  if ( !mf )
  {
    *sum = 0;
    *length = 0;
    return -1;
  }
  *sum = osd_fcrc( mf );
  *length = mf->filelength;
  osd_fclose( mf );

  return 0;
}

int osd_faccess( const char * filename, int filetype )
{
  struct MAMEpaths *thepath;
  char *suffix, found = 0;
  unsigned int i;
  char buffer[1024];

//  logerror( "Test access %s %d.", filename, filetype );

  switch( filetype )
  {
    case OSD_FILETYPE_ROM:
      thepath = globaloptions.rompath;
      suffix = "\0";
    break;
    case OSD_FILETYPE_SAMPLE:
      thepath = globaloptions.samplepath;
      suffix = "\0";
    break;

    case OSD_FILETYPE_NVRAM:
      thepath = globaloptions.statepath;
      suffix = ".nv";
    break;

    case OSD_FILETYPE_HIGHSCORE:
      thepath = globaloptions.statepath;
      suffix = ".hi";
    break;

    case OSD_FILETYPE_CONFIG:
      thepath = globaloptions.statepath;
      suffix = ".cfg";
    break;

    case OSD_FILETYPE_INPUTLOG:
      thepath = globaloptions.statepath;
      suffix = ".inp";
    break;

    case OSD_FILETYPE_STATE:
      thepath = globaloptions.statepath;
      suffix = ".sta";
    break;

    case OSD_FILETYPE_ARTWORK:
      thepath = globaloptions.artpath;
      suffix = ".png";
    break;

    case OSD_FILETYPE_MEMCARD:
      thepath = globaloptions.statepath;
      suffix = "\0";
    break;

    case OSD_FILETYPE_SCREENSHOT:
      thepath = globaloptions.capturepath;
      suffix = ".png";
    break;

    case OSD_FILETYPE_HIGHSCORE_DB:
    case OSD_FILETYPE_HISTORY:
    case OSD_FILETYPE_CHEAT:
      thepath = globaloptions.curpath;
      suffix = "\0";
    break;

    case OSD_FILETYPE_LANGUAGE:
      thepath = globaloptions.curpath;
      suffix = ".lng";
    break;

    default:
      logerror( "Attempted to access unknown file type: %d",
        filetype );
      return 0;
  }

  found = 0;

  for ( i = 0; i < thepath->numpaths; ++i )
  {
    // Look for path\filename.suffix
    // Look for path\filename.zip

    sprintf( buffer, "%s\\%s%s", thepath->paths[i],
     filename, suffix );

    if ( access( buffer, F_OK ) == 0 )
    {
      found = 1;
      break;
    }

    sprintf( buffer, "%s\\%s.zip", thepath->paths[i],
     filename );

    if ( access( buffer, R_OK ) == 0 )
    {
      found = 1;
      break;
    }
  }

//  if ( !found ) logerror( "Not found." );

  return found;
}

char * osd_fgets( char *s, int n, void *file )
{
  struct MAMEFILE *mf = (struct MAMEFILE *)file;

  CHECK_FILE_NULL( "osd_fgets", NULL );

  switch ( mf->filetype )
  {
    case 0:
      return fgets( s, n, mf->fp );
    case 1:
      if ( mf->fileposition + n < mf->filelength )
      {
        long int tmp = mf->fileposition;

        mf->fileposition += n;
        return strncpy( s, (char *)(mf->extradata + tmp), n );
      } else {
        strncpy( s, (char *)(mf->extradata + mf->fileposition),
          mf->filelength - mf->fileposition );
        mf->fileposition = mf->filelength;
        return s;
      }
    default:
      LOG_CORRUPT_POINTER( "osd_fgets" );
  }

  return NULL;
}

int osd_fgetc( void *file )
{
  struct MAMEFILE *mf = (struct MAMEFILE *)file;

  CHECK_FILE_NULL( "osd_fgetc", EOF );

  switch ( mf->filetype )
  {
    case 0:
      return fgetc( mf->fp );
    case 1:
      if ( mf->fileposition < mf->filelength )
      {
        mf->fileposition++;
        return mf->extradata[ mf->fileposition - 1 ];
      } else {
        return EOF;
      }
    default:
      LOG_CORRUPT_POINTER( "osd_fgetc" );
  }

  return EOF;
}

int osd_feof( void *file )
{
  struct MAMEFILE *mf = (struct MAMEFILE *)file;

  CHECK_FILE_NULL( "osd_feof", 1 );

  switch ( mf->filetype )
  {
    case 0:
      return feof( mf->fp );
    case 1:
      return (mf->fileposition >= mf->filelength);
    default:
      LOG_CORRUPT_POINTER( "osd_feof" );
  }
  return 1;
}

int osd_ungetc( int c, void *file )
{
  struct MAMEFILE *mf = (struct MAMEFILE *)file;

  CHECK_FILE_NULL( "osd_ungetc", EOF );

  switch ( mf->filetype )
  {
    case 0:
      return ungetc( c, mf->fp );
    case 1:
      return EOF;
    default:
      LOG_CORRUPT_POINTER( "osd_ungetc" );
  }

  return EOF;
}

void osd_fclose( void * file )
{
  struct MAMEFILE *mf = (struct MAMEFILE *)file;

  CHECK_FILE_NULL( "osd_fclose",  );

  switch ( mf->filetype )
  {
    case 0:
      fclose( mf->fp );
      return;
    case 1:
      free( mf->extradata );
      return;
    default:
      LOG_CORRUPT_POINTER( "osd_fclose" );
  }

  return;
}

