#define INCL_DOSPROCESS
#define INCL_DOSSEMAPHORES
#define INCL_DOSDEVICES
#define INCL_DOSDEVIOCTL
#define INCL_WININPUT
#define INCL_DOSMISC

#include "osdepend.h"
#include "dualmode.h"
#include "options.h"
#include "mame.h"
#include "driver.h"
#include "scancode.h"
#include "tmr0_ioc.h"
#include "xtramm.h"
#include "dart.h"
#include "frontend.h"
#include "profile.h"
#include <math.h>

struct GlobalOptions globaloptions;

char display_just_created = 0;
char frame_advance_mode = 0;
HEV frame_advance_sem = 0;
HEV run_game_sem = 0;
FILE *dbfp;

struct osd_bitmap *last_blitted_bmp = NULL;
int game_to_run = -1;

static ULONG last_audio_request_sizeS = 0;
static char audio_stereo_mult = 1;
static ULONG audio_buffer_num_frames, audio_buffer_cframe;
static ULONG audio_buffer_totallenS;
static ULONG *audio_buffer_frame_startsS, *audio_buffer_frame_lengthsS;

extern signed short *audio_cur_buffer;

extern char fast_exit;

#define MAX_FPS 120

typedef struct _MGL_PALETTE_ENTRY
{
  unsigned char blue, green, red, alpha;
} MGL_PALETTE_ENTRY;

typedef struct _MGL_SERVER_COLORS_SET_PACKET
{
  HQUEUE client_queue;            // INPUT - obtained by previous server call
  HEV client_wakeup;              // INPUT - obtained by previous server call
  int reserved[9];                // RESERVED
  MGL_PALETTE_ENTRY colors[256];  // INPUT
} MGL_SERVER_COLORS_SET_PACKET;

#define maker565( r, g, b ) (unsigned short) (((unsigned short)(b)>>3) \
 | ((unsigned short)(g)&0x00fc)<<3 | ((unsigned short)(r)&0x00f8)<<8)
// Make a 16 bit color value out of R G & B components

static char modifiable_palette = 0;

struct vidmode_descrip default_modes[] =
{
  { 256, 256, 8 },
  { 256, 256, 16 },
  { 320, 200, 8 },
  { 320, 200, 16 },
  { 320, 240, 8 },
  { 320, 240, 16 },
  { 320, 400, 8 },
  { 320, 400, 16 },
  { 640, 480, 8 },
  { 640, 480, 16 },
  { 800, 600, 8 },
  { 800, 600, 16 },
  { 1024, 768, 8 },
  { 1024, 768, 16 },
  { 1280, 1024, 8 },
  { 1280, 1024, 16 },
  { 0, 0, 0 }
};

struct vidmode_descrip *MAME_modes = default_modes;

struct VideoState
{
  HFILE kbd;
  int fps;
  float gamma;
  int bright;
  ULONG curframe;
  int attributes, screen_colors;
  char skipnextframe, framesskipped;
  ULONG rapidframe;
} vidstate;

struct KeyboardInfo keylist[] =
{
	{ "A",		KB_A,				KEYCODE_A },
	{ "B",		KB_B,				KEYCODE_B },
	{ "C",		KB_C,				KEYCODE_C },
	{ "D",		KB_D,				KEYCODE_D },
	{ "E",		KB_E,				KEYCODE_E },
	{ "F",		KB_F,				KEYCODE_F },
	{ "G",		KB_G,				KEYCODE_G },
	{ "H",		KB_H,				KEYCODE_H },
	{ "I",		KB_I,				KEYCODE_I },
	{ "J",		KB_J,				KEYCODE_J },
	{ "K",		KB_K,				KEYCODE_K },
	{ "L",		KB_L,				KEYCODE_L },
	{ "M",		KB_M,				KEYCODE_M },
	{ "N",		KB_N,				KEYCODE_N },
	{ "O",		KB_O,				KEYCODE_O },
	{ "P",		KB_P,				KEYCODE_P },
	{ "Q",		KB_Q,				KEYCODE_Q },
	{ "R",		KB_R,				KEYCODE_R },
	{ "S",		KB_S,				KEYCODE_S },
	{ "T",		KB_T,				KEYCODE_T },
	{ "U",		KB_U,				KEYCODE_U },
	{ "V",		KB_V,				KEYCODE_V },
	{ "W",		KB_W,				KEYCODE_W },
	{ "X",		KB_X,				KEYCODE_X },
	{ "Y",		KB_Y,				KEYCODE_Y },
	{ "Z",		KB_Z,				KEYCODE_Z },
	{ "0",		KB_0,				KEYCODE_0 },
	{ "1",		KB_1,				KEYCODE_1 },
	{ "2",		KB_2,				KEYCODE_2 },
	{ "3",		KB_3,				KEYCODE_3 },
	{ "4",		KB_4,				KEYCODE_4 },
	{ "5",		KB_5,				KEYCODE_5 },
	{ "6",		KB_6,				KEYCODE_6 },
	{ "7",		KB_7,				KEYCODE_7 },
	{ "8",		KB_8,				KEYCODE_8 },
	{ "9",		KB_9,				KEYCODE_9 },
	{ "0 PAD",	KB_padInsert,			KEYCODE_0_PAD },
	{ "1 PAD",	KB_padEnd,			KEYCODE_1_PAD },
	{ "2 PAD",	KB_padDown,			KEYCODE_2_PAD },
	{ "3 PAD",	KB_padPageDown,			KEYCODE_3_PAD },
	{ "4 PAD",	KB_padLeft,			KEYCODE_4_PAD },
	{ "5 PAD",	KB_padCenter,			KEYCODE_5_PAD },
	{ "6 PAD",	KB_padRight,			KEYCODE_6_PAD },
	{ "7 PAD",	KB_padHome,			KEYCODE_7_PAD },
	{ "8 PAD",	KB_padUp,			KEYCODE_8_PAD },
	{ "9 PAD",	KB_padPageUp,			KEYCODE_9_PAD },
	{ "F1",		KB_F1,				KEYCODE_F1 },
	{ "F2",		KB_F2,				KEYCODE_F2 },
	{ "F3",		KB_F3,				KEYCODE_F3 },
	{ "F4",		KB_F4,				KEYCODE_F4 },
	{ "F5",		KB_F5,				KEYCODE_F5 },
	{ "F6",		KB_F6,				KEYCODE_F6 },
	{ "F7",		KB_F7,				KEYCODE_F7 },
	{ "F8",		KB_F8,				KEYCODE_F8 },
	{ "F9",		KB_F9,				KEYCODE_F9 },
	{ "F10",	KB_F10,			KEYCODE_F10 },
	{ "F11",	KB_F11,			KEYCODE_F11 },
	{ "F12",	KB_F12,			KEYCODE_F12 },
	{ "ESC",	KB_esc,			KEYCODE_ESC },
	{ "~",		KB_tilde,		KEYCODE_TILDE },
	{ "-",          KB_minus,          KEYCODE_MINUS },
	{ "=",          KB_equals,         KEYCODE_EQUALS },
	{ "BKSPACE",	KB_backspace,	   KEYCODE_BACKSPACE },
	{ "TAB",	KB_tab,	           KEYCODE_TAB },
	{ "[",          KB_leftSquareBrace,      KEYCODE_OPENBRACE },
	{ "]",          KB_rightSquareBrace,     KEYCODE_CLOSEBRACE },
	{ "ENTER",	KB_enter,	   KEYCODE_ENTER },
	{ ";",          KB_semicolon,      KEYCODE_COLON },
	{ "'",          KB_apostrophe,     KEYCODE_QUOTE },
	{ "\\",         KB_backSlash,      KEYCODE_BACKSLASH },
	{ ",",          KB_comma,          KEYCODE_COMMA },
	{ "<",          KB_comma,          KEYCODE_BACKSLASH2 }, // ????
	{ ".",          KB_period,         KEYCODE_STOP },
	{ "/",          KB_divide,         KEYCODE_SLASH },
	{ "SPACE",	KB_space,	   KEYCODE_SPACE },
	{ "INS",	KB_insert,			KEYCODE_INSERT },
	{ "DEL",	KB_delete,			KEYCODE_DEL },
	{ "HOME",	KB_home,			KEYCODE_HOME },
	{ "END",	KB_end,			KEYCODE_END },
	{ "PGUP",	KB_pageUp,			KEYCODE_PGUP },
	{ "PGDN",	KB_pageDown,			KEYCODE_PGDN },
	{ "LEFT",	KB_left,			KEYCODE_LEFT },
	{ "RIGHT",	KB_right,			KEYCODE_RIGHT },
	{ "UP",		KB_up,				KEYCODE_UP },
	{ "DOWN",	KB_down,			KEYCODE_DOWN },
	{ "/ PAD",      KB_padDivide,      KEYCODE_SLASH_PAD },
	{ "* PAD",      KB_padTimes,       KEYCODE_ASTERISK },
	{ "- PAD",      KB_padMinus,       KEYCODE_MINUS_PAD },
	{ "+ PAD",      KB_padPlus,        KEYCODE_PLUS_PAD },
	{ ". PAD",      KB_padDelete,      KEYCODE_DEL_PAD },
	{ "ENTER PAD",  KB_padEnter,       KEYCODE_ENTER_PAD },
	{ "PRTSCR",     KB_sysReq,         KEYCODE_PRTSCR },
//	{ "PAUSE",      KB_F12,            KEYCODE_PAUSE }, // No pause in MGL yet
	{ "L SHIFT",	KB_leftShift,			KEYCODE_LSHIFT },
	{ "R SHIFT",	KB_rightShift,			KEYCODE_RSHIFT },
	{ "L CTRL",	KB_leftCtrl,       		KEYCODE_LCONTROL },
	{ "R CTRL",	KB_rightCtrl,      		KEYCODE_RCONTROL },
	{ "L ALT",	KB_leftAlt,			KEYCODE_LALT },
	{ "R ALT",	KB_rightAlt,			KEYCODE_RALT },
	{ "SCRLOCK",    KB_scrollLock,                  KEYCODE_SCRLOCK },
	{ "NUMLOCK",    KB_numLock,            KEYCODE_NUMLOCK },
	{ "CAPSLOCK",   KB_capsLock,           KEYCODE_CAPSLOCK },
	{ 0, 0, 0 }	/* end of table */
};

char rapidkeys[ 256 ] = { 0 };

struct JoystickInfo joylist[] =
{
	{ "Mouse B1",   0x4001, JOYCODE_1_BUTTON1 },
	{ "Mouse B2",   0x4002, JOYCODE_1_BUTTON2 },
	{ "Mouse B3",   0x4004, JOYCODE_1_BUTTON3 },
        // Hard-code mouse as player 1 for now.
	{ 0, 0, 0 }	/* end of table */
};

void InputHandler( InputEventType, short, short );

extern MRESULT WindowProc( HWND, ULONG, MPARAM, MPARAM );
extern void WindowInit( HWND );
extern void WindowCommandProc( HWND, ULONG );

struct MGLSC_event_handlers eventhandler =
{
  WindowProc,
  // Additional window procedure
  WindowInit,
  // Window initializer
  InputHandler,
  // Input event handler
  NULL,
  // Window closed event handler
  WindowCommandProc,
  // PM menu action processing
};

extern void myerrorlog( const char *text, ... );
// In frontend.c

char keymap[ DEFINED_KEYS ] = { 0 };
int last_keyup = 0;
int mousex = 0, mousey = 0, mousebuttons = 0;

void InputHandler( InputEventType event, short data1, short data2 )
{
  extern char isfullscreen;

  switch ( event )
  {
    case MGLSC_KEYBOARD_MAKE:
      keymap[data1] = 1;
      if ( data1 == KB_F11 )
      {
        if ( !MGLSC_clientState()->isFullScreen )
        {
          if ( !MGLSC_clientState()->server_pid && Machine->sample_rate )
          {
            isfullscreen = 1;

            WinSendMsg( MGLSC_clientState()->clientwin, WM_TOGGLEFS, 0, 0 );
            // Need to do this otherwise we lock for some stupid reason.
            // Audio threads need to be shut down when kicking full screen
            //  for the first time.

            isfullscreen = 0;
            return;
          }
        }

        WinPostMsg( MGLSC_clientState()->clientwin, WM_TOGGLEFS, 0, 0 );
        return;
      }
    break;
    case MGLSC_KEYBOARD_BREAK:
      keymap[data1] = 0;
      last_keyup = data1;

      if ( data1 == KB_padEnter )
      {
        frame_advance_mode = 1 - frame_advance_mode;
        if ( !frame_advance_mode )
        {
          DosPostEventSem( frame_advance_sem );
          if ( globaloptions.enablemouse && globaloptions.mousegrab )
          {
            MGLSC_mouseGrab( 1 );
          }
        } else {
          MGLSC_mouseGrab( 0 );
        }
      } else if ( frame_advance_mode && data1 == KB_padPlus )
      {
        DosPostEventSem( frame_advance_sem );
      } else if ( frame_advance_mode && data1 == KB_F12 )
      {
        // Allow single frame screen capture
        WinPostMsg( MGLSC_clientState()->clientwin, WM_CLIPCAPTURE, 0, 0 );
      }
    break;
    case MGLSC_MOUSE_BUTTON_PRESS:
      mousebuttons = data1;
    break;
    case MGLSC_MOUSE_BUTTON_RELEASE:
      mousebuttons ^= data1;
    break;
    case MGLSC_MOUSE_MOVE:
      mousex += data1;
      mousey += data2;
    break;
    default:
  }
}

void writeKeyboardLEDs( char caps, char num, char scroll )
{
  static char lastcaps = 0, lastnum = 0, lastscroll = 0;

  ULONG ulAction, ulLength;
  SHIFTSTATE ss;
  BYTE KeyState[257];

  if ( (lastcaps == caps) && (lastnum == num) && (lastscroll == scroll) )
    return;

  lastcaps = caps;
  lastnum = num;
  lastscroll = scroll;

  if ( !vidstate.kbd )
  {
    myerrorlog( "ERROR!:  Keyboard device was closed unexpectedly." );
    globaloptions.simled = 0;
    return;
  }

  ulAction = 0;
  ulLength = sizeof(ss);
  DosDevIOCtl( vidstate.kbd, IOCTL_KEYBOARD, KBD_GETSHIFTSTATE, 0, 0,
   &ulAction, &ss, sizeof(ss), &ulLength);

  WinSetKeyboardStateTable( HWND_DESKTOP, KeyState, FALSE );

  if ( caps )
  {
    KeyState[ VK_CAPSLOCK ] |= 1;
    ss.fsState |= CAPSLOCK_ON;
  } else {
    KeyState[ VK_CAPSLOCK ] &= ~1;
    ss.fsState &= ~CAPSLOCK_ON;
  }
  if ( num )
  {
    KeyState[ VK_NUMLOCK ] |= 1;
    ss.fsState |= NUMLOCK_ON;
  } else {
    KeyState[ VK_NUMLOCK ] &= ~1;
    ss.fsState &= ~NUMLOCK_ON;
  }
  if ( scroll )
  {
    KeyState[ VK_SCRLLOCK ] |= 1;
    ss.fsState |= SCROLLLOCK_ON;
  } else {
    KeyState[ VK_SCRLLOCK ] &= ~1;
    ss.fsState &= ~SCROLLLOCK_ON;
  }

  WinSetKeyboardStateTable( HWND_DESKTOP, KeyState, TRUE );

  ulAction = sizeof(ss);
  ulLength = 0;
  DosDevIOCtl( vidstate.kbd, IOCTL_KEYBOARD, KBD_SETSHIFTSTATE, &ss,
   sizeof(ss), &ulAction, 0, 0, &ulLength);
}

void useKeyboardLEDs( char yesno )
{
  if ( yesno )
  {
    ULONG action;
    ULONG ulOpenFlag = OPEN_ACTION_OPEN_IF_EXISTS;
    ULONG ulOpenMode = OPEN_FLAGS_FAIL_ON_ERROR | OPEN_SHARE_DENYNONE |
                       OPEN_ACCESS_READONLY;

    if ( vidstate.kbd ) return;

    if ( DosOpen("KBD$", &(vidstate.kbd), &action, 0L, 0, ulOpenFlag,
     ulOpenMode, NULL ) )
    {
      myerrorlog( "Could not open the keyboard device to change LEDs.\n" );
      globaloptions.simled = 0;
    }
    myerrorlog( "Keyboard device driver opened." );
  } else {
    if ( !vidstate.kbd ) return;
    WinPostMsg( MGLSC_clientState()->clientwin, WM_KEYLEDS,
     MPFROM2SHORT( 0, 0 ), MPFROMSHORT( 0 ) );
    DosClose( vidstate.kbd );
    vidstate.kbd = 0;
    myerrorlog( "Keyboard device driver closed." );
  }
}

int osd_init( void )
{
  if ( !globaloptions.enableaudio )
  {
    Machine->sample_rate = 0;  // disable sound
    options.samplerate = 0;
    options.use_samples = 0;
  }

  if ( globaloptions.enablemouse && globaloptions.mousegrab )
  {
    MGLSC_mouseGrab( 1 );
    MGLSC_setMouseModeRelative( 1 );
  }

  if ( globaloptions.simled )
  {
    useKeyboardLEDs( 1 );
  }

  last_blitted_bmp = NULL;
  return 0;
}

void osd_exit( void )
{
  extern void ss_deregister_all_funcs( void );
  MGLSC_mouseGrab( 0 );
  useKeyboardLEDs( 0 );
  ss_deregister_all_funcs();
  WinPostMsg( MGLSC_clientState()->clientwin, WM_ENDGAME, 0, 0 );
  last_blitted_bmp = NULL;
}

#define MAME_SAFETY 16

int osd_create_display( int width, int height, int depth, int fps, int attributes,
 int orientation )
{
  myerrorlog( "Create display %d x %d %dbpp, %d fps.", width, height, depth, fps );

  vidstate.attributes = attributes;
  display_just_created = 1;

  vidstate.fps = fps;
  vidstate.bright = 100;
  vidstate.gamma = 1.0;

  MGLSC_setFrameRate( fps );

  return 0;
}

void osd_close_display( void )
{
  unsigned short *vidbuffer;
  int i, j;

  WinPostMsg( MGLSC_clientState()->clientwin, WM_SETWINSIZE,
   MPFROM2SHORT( 1000, 1000 ), 0 );

  MGLSC_reinit( 640, 480, 16, FORMAT_DIRECTMAP | FORMAT_RGB, 0, 0 );
  vidbuffer = MGLSC_clientState()->vidbuffer;

  for ( i=0; i<480; ++i )
  {
    for ( j=0; j<640; ++j )
    {
      vidbuffer[ (i*640)+j ] = maker565( (i*255) / 480,
       ((i+j)*255 / (640 + 480)), (j*255) / 640 );
    }
  }

  MGLSC_setViewPort( 0, 0, 639, 479, 1 );
  WinPostMsg( MGLSC_clientState()->clientwin, WM_BLITFRAME, 0, 0 );
}

struct osd_bitmap *osd_alloc_bitmap( int width, int height, int depth)
{
  struct osd_bitmap *bmp;
  int i, rowtot = height + (MAME_SAFETY << 1);
  unsigned char *thebitmap = NULL;

  bmp = (struct osd_bitmap *) malloc( sizeof( struct osd_bitmap ) );
  if ( !bmp ) return NULL;

  bmp->width = width;
  bmp->height = height;
  bmp->depth = depth;

  myerrorlog( "Allocating %s bitmap:  %d x %d %dbpp.  Handle %lx.",
    display_just_created ? "screen" : "backbuffer", width, height, depth, bmp );

  if ( display_just_created )
  {
    // This first call is to allocate the actual backbuffer (I hope)

    display_just_created = 0;

    if ( depth > 8 )
    {
      MGLSC_reinit( width + (MAME_SAFETY*2), height + (MAME_SAFETY*2), depth,
       FORMAT_DIRECTMAP | FORMAT_RGB, 0, 0 );
    } else {
      MGLSC_reinit( width + (MAME_SAFETY*2), height + (MAME_SAFETY*2), depth,
       FORMAT_LOOKUPTABLE, 0, 0 );
    }

    last_blitted_bmp = bmp;
    
    bmp->_private = NULL;
    thebitmap = MGLSC_clientState()->vidbuffer;
    // This bitmap is managed by DualMode
  } else {
    bmp->_private = malloc( rowtot * (width + (MAME_SAFETY << 1)) * (depth / 8));
    thebitmap = bmp->_private;

    if ( !bmp->_private )
    {
      free( bmp );
      return NULL;
    }
  }

  bmp->line = malloc(rowtot * sizeof(unsigned char *));

  if ( !bmp->line )
  {
    if ( bmp->_private ) free( bmp->_private );
    free( bmp );
    return NULL;
  }

  for ( i=0; i<rowtot; ++i )
  {
    bmp->line[i] = &( thebitmap[(i * (width + (MAME_SAFETY << 1)) * (depth>>3)) + 
      (MAME_SAFETY * (depth>>3))] );
  }

  bmp->line += MAME_SAFETY;
  // YUCK!  How sloppy!  Guess I can't even be sure that the stupid
  // MAME core isn't going to index into this array with a negative
  // number.  I don't write 'em.  I just port 'em.  ;-)

  return bmp;
}

void osd_free_bitmap( struct osd_bitmap *bitmap )
{
  if ( last_blitted_bmp == bitmap )
  {
    last_blitted_bmp = NULL;
  }

  if ( !bitmap )
  {
    myerrorlog( "ERROR!:  Attempted to free an unallocated bitmap (handle NULL)." );
    return;
  }

  if ( !bitmap->_private )
  {
    myerrorlog( "MAME core freed the screen buffer (handle %lx).", bitmap );
  } else {
    free( bitmap->_private );
  }

  if ( !bitmap->line )
  {
    myerrorlog( "ERROR!:  Attempted to free a bitmap that has no line array." );
  } else {
    bitmap->line -= MAME_SAFETY;
    // Un-yuckify.

    free( bitmap->line );
  }

  myerrorlog( "Deallocating bitmap handle %lx.", bitmap );

  free( bitmap );
}

int osd_skip_this_frame( void )
{
  if ( globaloptions.autoskip )
  {
    vidstate.skipnextframe = MGLSC_shouldSkipFrame();
  } else {
    if ( vidstate.curframe % (globaloptions.mskip+1) == 0 )
      vidstate.skipnextframe = 0;
    else vidstate.skipnextframe = 1;
  }

  return vidstate.skipnextframe;
}

void osd_mark_dirty( int xmin, int ymin, int xmax, int ymax )
{

}

void osd_update_video_and_audio( struct osd_bitmap *bitmap,
 struct osd_bitmap *debug_bitmap, int led_status )
{
  static ULONG junk;

  MGLSC_flushUserInput();

  if ( globaloptions.simled )
  {
    WinPostMsg( MGLSC_clientState()->clientwin, WM_KEYLEDS,
     MPFROM2SHORT( (led_status & 1), (led_status & 2)),
     MPFROMSHORT( (led_status & 4) ) );
  }

  if ( bitmap->_private && last_blitted_bmp )
  {
    unsigned char **tmpline;

    // Yucky hack
    memcpy( MGLSC_clientState()->vidbuffer, bitmap->_private,
     (bitmap->height + (MAME_SAFETY << 1)) * 
     (bitmap->width + (MAME_SAFETY << 1)) * (bitmap->depth / 8) );
    tmpline = last_blitted_bmp->line;
    last_blitted_bmp->line = bitmap->line;
    bitmap->line = tmpline;
    last_blitted_bmp->_private = bitmap->_private;
    bitmap->_private = NULL;

    last_blitted_bmp = bitmap;
    // Perform a switcheroo and hope the MAME core is too dumb to notice
  }

  if ( frame_advance_mode )
  {
    DosResetEventSem( MGLSC_clientState()->framesync_sem, &junk );
    WinPostMsg( MGLSC_clientState()->clientwin, WM_BLITFRAME, 0, 0 );
    DosWaitEventSem( MGLSC_clientState()->framesync_sem, 0xffffffff );

    // Blit the single frame at the user's explicit request

    if ( Machine->sample_rate )
    {
      int i;

      for ( i=0; i<SoundSys.ulNumBuffers; ++i )
      {
        memset( SoundSys.MixBuffers[i].pBuffer, 0,
          audio_buffer_totallenS * 2 * audio_stereo_mult );
      }
    }

    DosResetEventSem( frame_advance_sem, &junk );
    DosWaitEventSem( frame_advance_sem, 0xffffffff );

    vidstate.curframe = 0;
    // Make sure we don't skip excessively when we leave single frame mode

    return;
  }

  if ( !vidstate.skipnextframe )
  {
    vidstate.framesskipped = 0;

    MGLSC_blockUntilFrameReady();

    DosResetEventSem( MGLSC_clientState()->framesync_sem, &junk );
    WinPostMsg( MGLSC_clientState()->clientwin, WM_BLITFRAME, 0, 0 );
    DosWaitEventSem( MGLSC_clientState()->framesync_sem, SEM_INDEFINITE_WAIT );
  } else vidstate.framesskipped++;

  MGLSC_advanceFrame( vidstate.skipnextframe );

  vidstate.curframe++;
  if ( vidstate.curframe >= vidstate.rapidframe )
  {
    vidstate.rapidframe = (vidstate.curframe + globaloptions.rapidrate) %
     vidstate.fps;
  }

  if ( vidstate.curframe >= vidstate.fps )
  {
    vidstate.curframe = 0;
  }
}

void osd_set_visible_area(int min_x,int max_x,int min_y,int max_y)
{
  myerrorlog( "Viewport set to:  %d, %d -> %d, %d", min_x, min_y, max_x, max_y );

  MGLSC_setViewPort( min_x + MAME_SAFETY, min_y + MAME_SAFETY,
   max_x + MAME_SAFETY, max_y + MAME_SAFETY, 1 );

  set_ui_visarea( min_x, min_y, max_x, max_y );
}

int osd_display_loading_rom_message( const char *name, int current, int total)
{
  extern char abortload;
  // in frontend.c

  char *dup = NULL;

  if ( name )
  {
    dup = malloc( strlen(name) + 1 );
    strcpy( dup, name );
  }

  WinPostMsg( MGLSC_clientState()->clientwin, WM_LOADPROGRESS,
    MPFROMP(dup), MPFROM2SHORT( current, total ) );
  // Window procedure will free this memory

  return abortload;
  // Returning non-0 aborts the loading of ROMs
}

void osd_save_snapshot( struct osd_bitmap *bitmap )
{
  WinPostMsg( MGLSC_clientState()->clientwin, WM_CLIPCAPTURE, 0, 0 );
}

void osd_joystick_start_calibration( void )
{

}

void osd_joystick_calibrate( void )
{

}

const char *osd_joystick_calibrate_next( void )
{
  return NULL;
}

void osd_joystick_end_calibration( void )
{

}

int osd_joystick_needs_calibration( void )
{
  return 0;
}

void osd_analogjoy_read( int player, int *analog_x, int *analog_y )
{
  // -128 to 128
}

void osd_trak_read( int player, int *deltax, int *deltay )
{
  if ( globaloptions.enablemouse )
  {
    *deltax = mousex;  mousex = 0;
    *deltay = mousey;  mousey = 0;
  } else {
    *deltax = 0;
    *deltay = 0;
  }
}

void osd_sound_enable( int enable )
{

}

#define NUM_AUDIO_FRAMES globaloptions.audinterleave

int osd_start_audio_stream( int stereo )
{
  ULONG i, bufferlen;

  last_audio_request_sizeS = 0;
  audio_buffer_cframe = 0;
  audio_buffer_num_frames = 0;

  if ( !Machine->sample_rate )
  {
    return 1;
  }

  audio_stereo_mult = (stereo ? 2 : 1);

  audio_buffer_totallenS =
   (NUM_AUDIO_FRAMES * Machine->sample_rate) / vidstate.fps;
  bufferlen = audio_buffer_totallenS * 2 * audio_stereo_mult;
  audio_buffer_num_frames = NUM_AUDIO_FRAMES;
  audio_buffer_cframe = 0;
  audio_buffer_frame_startsS = (ULONG *)malloc( NUM_AUDIO_FRAMES * sizeof(ULONG *) );

  for ( i=0; i < audio_buffer_num_frames; ++i )
  {
    audio_buffer_frame_startsS[i] = (audio_buffer_totallenS * i) /
     audio_buffer_num_frames;
  }
  audio_buffer_frame_lengthsS = (ULONG *)malloc( NUM_AUDIO_FRAMES * sizeof(ULONG *) );
  for ( i=0; i < audio_buffer_num_frames - 1; ++i )
  {
    audio_buffer_frame_lengthsS[i] = audio_buffer_frame_startsS[i+1] -
     audio_buffer_frame_startsS[i];
  }
  audio_buffer_frame_lengthsS[i] = 
   audio_buffer_totallenS - audio_buffer_frame_startsS[i];

  if ( DART_init( Machine->sample_rate, &bufferlen, 16, stereo,
        globaloptions.shareaudio ) )
  {
    myerrorlog( "Audio initialization failed." );
    Machine->sample_rate = 0;
    options.samplerate = 0;
    options.use_samples = 0;
    free( audio_buffer_frame_startsS );
    free( audio_buffer_frame_lengthsS );
    audio_buffer_num_frames = 0;
    return 1;
  }

  if ( bufferlen != audio_buffer_totallenS * 2 * audio_stereo_mult )
  {
    myerrorlog( "DART didn't like my audio buffer size of %ld, requesting that I use %ld instead.  Sound disabled.\n", 
     audio_buffer_totallenS * 2 * audio_stereo_mult, bufferlen );
    options.samplerate = 0;
    options.use_samples = 0;
    free( audio_buffer_frame_startsS );
    free( audio_buffer_frame_lengthsS );
    audio_buffer_num_frames = 0;
    Machine->sample_rate = 0;
    DART_shutdown();
    return 1;
  }

  last_audio_request_sizeS = audio_buffer_frame_lengthsS[0];

  return last_audio_request_sizeS;
}

int osd_update_audio_stream( INT16 *buffer )
{
  ULONG junk;

  if ( !audio_buffer_num_frames ) return 0;

  memcpy( 
   &(audio_cur_buffer[ audio_buffer_frame_startsS[audio_buffer_cframe] * audio_stereo_mult ]),
   buffer, last_audio_request_sizeS * 2 * audio_stereo_mult );
  // Write the current audio "frame" to the current buffer.

  audio_buffer_cframe = (audio_buffer_cframe + 1) % audio_buffer_num_frames;
  // Advance the audio "frame".

  if ( !audio_buffer_cframe )
  {
    // If we've wrapped around back to 0, then we've completed a buffer.

    DosRequestMutexSem( SoundSys.semDARTMutex, SEM_INDEFINITE_WAIT );

    DosResetEventSem( SoundSys.semNeedsBuffer, &junk );
    DosPostEventSem( SoundSys.semHasBuffer );

    if ( SoundSys.ulBufferHead == SoundSys.ulBufferTail )
    {
      DosReleaseMutexSem( SoundSys.semDARTMutex );
      DosWaitEventSem( SoundSys.semNeedsBuffer, SEM_INDEFINITE_WAIT );
      DosRequestMutexSem( SoundSys.semDARTMutex, SEM_INDEFINITE_WAIT );
    }

    SoundSys.ulBufferHead = (SoundSys.ulBufferHead+1)%SoundSys.ulNumBuffers;
    // Advance the "write head" to the next buffer

    DosReleaseMutexSem( SoundSys.semDARTMutex );

    audio_cur_buffer = (signed short *)
     SoundSys.MixBuffers[SoundSys.ulBufferHead].pBuffer;
    // Point to the current write buffer.
  }

  last_audio_request_sizeS = audio_buffer_frame_lengthsS[ audio_buffer_cframe ];

  return audio_buffer_frame_lengthsS[ audio_buffer_cframe ];
}

void osd_stop_audio_stream( void )
{
  DART_shutdown();
  
  free( audio_buffer_frame_lengthsS );
  free( audio_buffer_frame_startsS );
}

void osd_opl_control( int chip,int reg )
{

}

void osd_opl_write( int chip,int data )
{

}

int status_paused = 0;

void osd_pause( int paused )
{
  if ( (globaloptions.enablemouse && globaloptions.mousegrab) || paused )
  {
    MGLSC_mouseGrab( !paused );
  }

  status_paused = paused;

  if ( Machine->sample_rate && SoundSys.ulNumBuffers )
  {
    int i;

    for ( i=0; i<SoundSys.ulNumBuffers; ++i )
    {
      memset( SoundSys.MixBuffers[i].pBuffer, 0,
        audio_buffer_totallenS * 2 * audio_stereo_mult );
    }
  }
}

void osd_set_mastervolume( int attenuation )
{

}

int osd_get_mastervolume( void )
{
  return 1;
}

void osd_set_gamma( float _gamma )
{
  vidstate.gamma = _gamma;
}

float osd_get_gamma( void )
{
  return vidstate.gamma;
}

void osd_set_brightness( int brightness )
{
  vidstate.bright = brightness;
}

int osd_get_brightness( void )
{
  return vidstate.bright;
}


#define makecol( r, g, b ) (unsigned short) (((unsigned short)(b)>>3) | ((unsigned short)(g)&0x00fc)<<3 | ((unsigned short)(r)&0x00f8)<<8)

int osd_allocate_colors(unsigned int totalcolors, const UINT8 *palette,
 UINT32 *pens, int modifiable, const UINT8 *debug_palette, UINT32 *debug_pens)
{
	int i;
  MGL_SERVER_COLORS_SET_PACKET *cs = (MGL_SERVER_COLORS_SET_PACKET *)
   MGLSC_clientState()->shared_packet;

  myerrorlog( "MAME core requested %d colors (%s).", totalcolors,
   modifiable ? "dynamic" : "static" );

	modifiable_palette = modifiable;

	if (vidstate.attributes & VIDEO_RGB_DIRECT)
  {
  }

	vidstate.screen_colors = totalcolors;
	if ( MGLSC_clientState()->depth != 8)
		vidstate.screen_colors += 2;
	else vidstate.screen_colors = 256;

//	palette_16bit_lookup = malloc(vidstate.screen_colors * sizeof(palette_16bit_lookup[0]));

	if ( MGLSC_clientState()->depth != 8 && modifiable == 0)
	{
		int r,g,b;

		for (i = 0;i < totalcolors;i++)
		{
			r = 255 * vidstate.bright * pow(palette[3*i+0] / 255.0, 1 / vidstate.gamma) / 100;
			g = 255 * vidstate.bright * pow(palette[3*i+1] / 255.0, 1 / vidstate.gamma) / 100;
			b = 255 * vidstate.bright * pow(palette[3*i+2] / 255.0, 1 / vidstate.gamma) / 100;
			*pens++ = makecol(r,g,b);
		}

		Machine->uifont->colortable[0] = makecol(0x00,0x00,0x00);
		Machine->uifont->colortable[1] = makecol(0xff,0xff,0xff);
		Machine->uifont->colortable[2] = makecol(0xff,0xff,0xff);
		Machine->uifont->colortable[3] = makecol(0x00,0x00,0x00);

		if (debug_pens)
		{
			for (i = 0;i < DEBUGGER_TOTAL_COLORS;i++)
			{
				r = debug_palette[3*i+0];
				g = debug_palette[3*i+1];
				b = debug_palette[3*i+2];
				*debug_pens++ = makecol(r,g,b);
			}
		}
  }	else {
    if ( MGLSC_clientState()->depth == 8 )
    {
    	MGLSC_clientState()->palette_dirty = 1;

    	for ( i=0; i < 256; i++ )
    		cs->colors[i].red = cs->colors[i].green = cs->colors[i].blue = 0;

      myerrorlog( "Palette cleared.\n" );

  		if ( totalcolors >= 255 )
  		{
  			int bestblack,bestwhite;
		  	int bestblackscore,bestwhitescore;

  			bestblack = bestwhite = 0;
  			bestblackscore = 3*255*255;
		  	bestwhitescore = 0;
  			for (i = 0;i < totalcolors; i++)
		  	{
  				int r,g,b,score;

		  		r = palette[3*i+0];
				  g = palette[3*i+1];
  				b = palette[3*i+2];
		  		score = r*r + g*g + b*b;

  				if (score < bestblackscore)
		  		{
				  	bestblack = i;
  					bestblackscore = score;
		  		}
				  if (score > bestwhitescore)
  				{
		  			bestwhite = i;
				  	bestwhitescore = score;
  				}
		  	}

  			for ( i=0; i < totalcolors; i++ )
		  		pens[i] = i;

        myerrorlog( "Pens array has been initialized.\n" );

  			/* map black to pen 0, otherwise the screen border will not be black */
		  	pens[bestblack] = 0;
  			pens[0] = bestblack;

  			Machine->uifont->colortable[0] = pens[bestblack];
		  	Machine->uifont->colortable[1] = pens[bestwhite];
  			Machine->uifont->colortable[2] = pens[bestwhite];
		  	Machine->uifont->colortable[3] = pens[bestblack];
  		}	else {
  			/* reserve color 1 for the user interface text */
        cs->colors[1].red   = 0xff;
        cs->colors[1].green = 0xff;
        cs->colors[1].blue  = 0xff;
  			Machine->uifont->colortable[0] = 0;
  			Machine->uifont->colortable[1] = 1;
  			Machine->uifont->colortable[2] = 1;
  			Machine->uifont->colortable[3] = 0;

  			/* fill the palette starting from the end, so we mess up badly written */
  			/* drivers which don't go through Machine->pens[] */
        for ( i=0; i<totalcolors; i++ )
          pens[i] = (vidstate.screen_colors-1)-i;
      }

      for ( i=0; i < totalcolors; i++ )
      {
        cs->colors[pens[i]].red   = palette[(3*i)];
        cs->colors[pens[i]].green = palette[(3*i)+1];
        cs->colors[pens[i]].blue  = palette[(3*i)+2];
//      myerrorlog( "PEN %d:  %dR %dG %dB", i, cs->colors[pens[i]].red, cs->colors[pens[i]].green, cs->colors[pens[i]].blue );
      }

//      myerrorlog( "UI COLOR 0:  %dR %dG %dB", cs->colors[Machine->uifont->colortable[0]].red,
//        cs->colors[Machine->uifont->colortable[0]].green,
//        cs->colors[Machine->uifont->colortable[0]].blue );
//      myerrorlog( "UI COLOR 1:  %dR %dG %dB", cs->colors[Machine->uifont->colortable[1]].red,
//        cs->colors[Machine->uifont->colortable[1]].green,
//        cs->colors[Machine->uifont->colortable[1]].blue );

      myerrorlog( "8 bit palette has been set up.\n" );
    }

		if (debug_pens)
		{
			for (i = 0;i < DEBUGGER_TOTAL_COLORS;i++)
				debug_pens[i] = i;
		}
	}

  WinPostMsg( MGLSC_clientState()->clientwin, WM_BLITFRAME, 0, 0 );
  // Make sure palette change is acknowledged

	// dbg_palette = debug_palette;

	return 0;
}

void osd_modify_pen( int pen, unsigned char red, unsigned char green,
 unsigned char blue )
{
  if ( modifiable_palette && MGLSC_clientState()->depth == 8 )
  {
    MGL_SERVER_COLORS_SET_PACKET *cs = (MGL_SERVER_COLORS_SET_PACKET *)
     MGLSC_clientState()->shared_packet;

    cs->colors[pen].red = red;
    cs->colors[pen].green = green;
    cs->colors[pen].blue = blue;

  	MGLSC_clientState()->palette_dirty = 1;

//    myerrorlog( "SET PEN: %d = %dR %dG %dB", pen, red, green, blue );
  }
}

void osd_get_pen( int pen, unsigned char *red, unsigned char *green,
 unsigned char *blue )
{
  MGL_SERVER_COLORS_SET_PACKET *cs = (MGL_SERVER_COLORS_SET_PACKET *)
   MGLSC_clientState()->shared_packet;

  if ( MGLSC_clientState()->depth == 8 )
  {
    *red   = cs->colors[pen].red;
    *green = cs->colors[pen].green;
    *blue  = cs->colors[pen].blue;
  }
//  myerrorlog( "GET PEN: %d (= %dR %dG %dB)", pen, *red, *green, *blue );
}

const struct KeyboardInfo *osd_get_key_list( void )
{
  return keylist;
}

const struct JoystickInfo *osd_get_joy_list( void )
{
  return joylist;
}

// Rapidkeys states
// 0 = not a rapid key
// 1 = rapid key, but not pressed
// 2 = rapid key, pressed, active in first half
// 3 = rapid key, pressed, active in second half

int osd_is_key_pressed( int keycode )
{
  if ( keymap[ keycode ] == 0 )
  {
    if ( rapidkeys[ keycode ] > 1 )
    {
      rapidkeys[ keycode ] = 1;
    }
    return 0;
  }
  // Key is not currently being held down.
  // Reset rapid fire state if we have one.

  if ( rapidkeys[ keycode ] == 0 )
  {
    return keymap[ keycode ];
  }

  // Otherwise, key is held down.  Modify with rapid fire settings
  // if applicable.

  if ( ((vidstate.rapidframe - (globaloptions.rapidrate>>1)) % vidstate.fps)
        <= vidstate.curframe )
  {
    // Second half of rapid keys

    if ( rapidkeys[ keycode ] == 1 )
    {
      // Newly pressed rapid key
      rapidkeys[ keycode ] = 3;
      return 1;
    }

    return rapidkeys[ keycode ] == 3;
  }

  // First half

  if ( rapidkeys[ keycode ] == 1 )
  {
    // Newly pressed rapid key
    rapidkeys[ keycode ] = 2;
    return 1;
  }

  return rapidkeys[ keycode ] == 2;
}

int osd_readkey_unicode( int flush )
{
  return last_keyup;
}

int osd_is_joy_pressed( int joycode )
{
  if ( joycode & 0x4000 && globaloptions.enablemouse )
  {
    // Mouse
    return ( mousebuttons & (joycode & 7) );
  }
  return 0;
}

void osd_customize_inputport_defaults( struct ipd *defaults )
{

}

char *onlyonepath[1] = { "." };
char *defaultrompath[2] = { "roms", "." };
char *defaultstatepath[2] = { "cfg", "." };
char *defaultartpath[2] = { "artwork", "." };
char *defaultsamplepath[2] = { "samples", "." };

struct MAMEpaths p1 = { 2, defaultsamplepath, 1 };
struct MAMEpaths p2 = { 2, defaultrompath, 1 };
struct MAMEpaths p3 = { 2, defaultstatepath, 1 };
struct MAMEpaths p4 = { 2, defaultartpath, 1 };
struct MAMEpaths p5 = { 1, onlyonepath, 1 };

int main ( int argc, char *argv[] )
{
  unsigned short *vidbuffer;
  int i, j, game = -1;
  ULONG junk;

  dbfp = fopen( "debug.log", "wa" );

  memset( &globaloptions, 0, sizeof( struct GlobalOptions ) );
  memset( &options, 0, sizeof( struct GameOptions ) );

  MGLSC_init( 640, 480, 16, FORMAT_DIRECTMAP | FORMAT_RGB, 0, 0, 639, 479,
   &eventhandler, "MAME Main Window", "MAME for OS/2", MAME_modes,
   MGLSC_WINDOW_DECOR_TASKLIST | MGLSC_WINDOW_DECOR_MINMAX |
   MGLSC_WINDOW_DECOR_SYSMENU | MGLSC_WINDOW_DECOR_TITLEBAR |
   MGLSC_WINDOW_DECOR_MENU | MGLSC_WINDOW_DECOR_ICON,
   0 /* custom modes */, 0 /* joystick */ );

  vidbuffer = MGLSC_clientState()->vidbuffer;
  for ( i=0; i<480; ++i )
  {
    for ( j=0; j<640; ++j )
    {
      vidbuffer[ (i*640)+j ] = maker565( (i*255) / 480,
       ((i+j)*255 / (640 + 480)), (j*255) / 640 );
    }
  }

  WinPostMsg( MGLSC_clientState()->clientwin, WM_BLITFRAME, 0, 0 );

  globaloptions.samplepath = &p1;
  globaloptions.rompath = &p2;
  globaloptions.statepath = &p3;
  globaloptions.artpath = &p4;
  globaloptions.capturepath = &p5;
  globaloptions.curpath = &p5;
  // Set up default paths

  OpenProfile();
  LoadPathsFromProfile();
  LoadFrameRateOptionsFromProfile();
  LoadAudioOptionsFromProfile();
  LoadMGLOptionsFromProfile();
  LoadRapidKeysForGame( -1 );
  LoadInputOptionsFromProfile( -1 );
  CloseProfile();

  options.color_depth = 8;
  options.vector_width = 640;
  options.vector_height = 480;
  options.translucency = 1;
  options.use_artwork = 1;
  options.samplerate = (globaloptions.enableaudio ? globaloptions.audfreq : 0);
  options.use_samples = globaloptions.enableaudio;
  options.gui_host = 1;
  options.use_filter = 1;
  options.antialias = 1;
  options.beam = 0x00010000;

  i = 0;

  if ( argc > 1 )
  {
    if ( argv[1][0] == '@' )
    {
      while ( drivers[i] )
      {
        myerrorlog( "%d: %s\n", i, drivers[i]->name );
        ++i;
      }
      WinPostMsg( MGLSC_clientState()->clientwin, WM_CLOSE, 0, 0 );
      DosWaitThread( &(MGLSC_clientState()->pm_thread), DCWW_WAIT );
      return 1;
    }

    while ( drivers[i] )
    {
      if ( strcmp( drivers[i]->name, argv[1] ) == 0 ) { game = i; break; }
      ++i;
    }
    if ( game == -1 )
    {
      myerrorlog( "Game %s is not supported as such.", argv[1] );
    }
  }

  DosCreateEventSem( NULL, &frame_advance_sem, 0, 0 );
  // This semaphore is used for single frame advancing

  DosCreateEventSem( NULL, &run_game_sem, 0, 0 );
  // This semaphore is tells us when to fire off the MAME core

  DosCreateEventSem( NULL, &(MGLSC_clientState()->framesync_sem), 0, 0 );
  // This semaphore is posted by the DualMode library when a frame blit is complete

  MGLSC_setTimingMode( globaloptions.usetimer0 );
  MGLSC_setMaxFrameSkip( globaloptions.autoskip );

  if ( game != -1 )
  {
    OpenProfile();
    LoadAvailableGamesFromProfile();
    CloseProfile();
    // Make sure the availablegames array is initialized

    game_to_run = game;
    WinPostMsg( MGLSC_clientState()->clientwin, WM_STARTGAME, 0, 0 );

    if ( run_game( game ) )
    {
      WinPostMsg( MGLSC_clientState()->clientwin, WM_GAMEERROR, 0, 0 );
    }

    OpenProfile();
    ConfirmGameIsNotDangerous( game );
    CloseProfile();

    if ( fast_exit )
      WinPostMsg( MGLSC_clientState()->clientwin, WM_CLOSE, 0, 0 );
  }

  fast_exit = 1;

  while ( 1 )
  {
    DosResetEventSem( run_game_sem, &junk );
    DosWaitEventSem( run_game_sem, SEM_INDEFINITE_WAIT );

    if ( game_to_run == -1 )
    {
      break;
    }

    fast_exit = 0;

    WinPostMsg( MGLSC_clientState()->clientwin, WM_STARTGAME, 0, 0 );

    keymap[KB_esc] = 0;

    memset( &options, 0, sizeof( struct GameOptions ) );

    options.color_depth = 8;
    options.vector_width = 640;
    options.vector_height = 480;
    options.translucency = 1;
    options.use_artwork = 1;
    options.samplerate = (globaloptions.enableaudio ? globaloptions.audfreq : 0);
    options.use_samples = globaloptions.enableaudio;
    options.gui_host = 1;
    options.use_filter = 1;
    options.antialias = 1;
    options.beam = 0x00010000;

    display_just_created = 0;

    if ( run_game( game_to_run ) )
    {
      WinPostMsg( MGLSC_clientState()->clientwin, WM_GAMEERROR, 0, 0 );
    }

    OpenProfile();
    ConfirmGameIsNotDangerous( game_to_run );
    CloseProfile();

    if ( fast_exit )
      break;

    fast_exit = 1;
  }

  fast_exit = 1;

  MGLSC_shutdown();

  DosCloseEventSem( MGLSC_clientState()->framesync_sem );

  fclose( dbfp );

  return 0;
}

