#define INCL_WINSHELLDATA
#include <os2.h>
#include <malloc.h>
#include "profile.h"
#include "dualmode.h"
#include "driver.h"

static HINI profile = 0;

extern void myerrorlog( const char *text, ... );

static int prfref = 0;

void OpenProfile( void )
{
  if ( prfref )
  {
    prfref++;
    return;
  }

  profile = PrfOpenProfile(
   WinQueryAnchorBlock( MGLSC_clientState()->clientwin ), "MAME.INI" );
}

void CloseProfile( void )
{
  if ( !prfref ) return;

  prfref--;

  if ( !prfref )
  {
    PrfCloseProfile( profile );
    profile = 0;
  }
}

static void WRITEPATH( char *pathname_string, struct MAMEpaths *pathptr )
{
  char numstring[4];
  int i;

  PrfWriteProfileData( profile, pathname_string, "Number of directories",
   &(pathptr->numpaths), 1 );
  for ( i=0; i<pathptr->numpaths; ++i )
  {
    sprintf( numstring, "%d", i );
    PrfWriteProfileString( profile, pathname_string, numstring,
     pathptr->paths[i] );
  }
}  

void SavePathsToProfile( void )
{
  WRITEPATH( "ROM Paths", globaloptions.rompath );
  WRITEPATH( "Sample Paths", globaloptions.samplepath );
  WRITEPATH( "Artwork Paths", globaloptions.artpath );
  WRITEPATH( "Saved State Paths", globaloptions.statepath );
  WRITEPATH( "Capture Paths", globaloptions.capturepath );
}

static struct MAMEpaths * READPATH( char *pathname_string, struct MAMEpaths *pathptr )
{
  ULONG datalen;
  int i;
  char numdirs = 0, numstring[4];
  struct MAMEpaths *newpath;

  datalen = 1;
  numdirs = 0;

  PrfQueryProfileData( profile, pathname_string, "Number of directories",
    &numdirs, &datalen );

  newpath = pathptr;

  if ( numdirs )
  {
    if ( !pathptr->isdefault )
    {
      for ( i=0; i<pathptr->numpaths; ++i )
      {
        free( pathptr->paths[i] );
      }
      free( pathptr->paths );
      free( pathptr );
    }

    newpath = (struct MAMEpaths *)malloc( sizeof( struct MAMEpaths ) );
    newpath->paths = (char **) malloc( numdirs * sizeof( char * ) );

    newpath->numpaths = numdirs;
    newpath->isdefault = 0;

    for ( i=0; i<numdirs; ++i )
    {
      sprintf( numstring, "%d", i );
      datalen = 0;
      PrfQueryProfileSize( profile, pathname_string, numstring, &datalen );
      newpath->paths[i] = (char *) malloc( datalen );
      PrfQueryProfileData( profile, pathname_string, numstring, newpath->paths[i],
        &datalen );
    }
  }

  return newpath;
}

void LoadPathsFromProfile( void )
{
  globaloptions.rompath     = READPATH( "ROM Paths", globaloptions.rompath );
  globaloptions.samplepath  = READPATH( "Sample Paths", globaloptions.samplepath );
  globaloptions.artpath     = READPATH( "Artwork Paths", globaloptions.artpath );
  globaloptions.statepath   = READPATH( "Saved State Paths", globaloptions.statepath );
  globaloptions.capturepath = READPATH( "Capture Paths", globaloptions.capturepath );
}

char *availabledrivers = NULL;
int totaldrivers = 0;

#define DRIVER_FLAGS_AVAILABLE 1
#define DRIVER_FLAGS_FAVORITE  2

void LoadAvailableGamesFromProfile( void )
{
  int i;
  char numstring[10];
  ULONG datalen;

  if ( !availabledrivers )
  {
    for ( i=0; drivers[i]; ++i );
    // count the total number of drivers

    totaldrivers = i;
    availabledrivers = (char *) malloc( totaldrivers );

    for ( i=0; i < totaldrivers; ++i ) availabledrivers[i] = 0;
  }

  for ( i=0; drivers[i]; ++i )
  {
    sprintf( numstring, "%d", i );
    datalen = 0;
    PrfQueryProfileSize( profile, "Quick List", numstring, &datalen );

    if ( datalen )
    {
      availabledrivers[ i ] |= DRIVER_FLAGS_AVAILABLE;

      datalen = 0;
      PrfQueryProfileSize( profile, "Potentially Dangerous", numstring,
       &datalen );

      if ( datalen )
      {
        availabledrivers[ i ] |= DRIVER_FLAGS_DANGEROUS;
      }
    }
  }
}

void LoadFavoriteGamesFromProfile( void )
{
  int i;
  char numstring[10];
  ULONG datalen;

  for ( i=0; i < totaldrivers; ++i )
  {
    sprintf( numstring, "%d", i );
    datalen = 0;
    PrfQueryProfileSize( profile, "Favorite List", numstring, &datalen );

    if ( datalen )
    {
      availabledrivers[ i ] |= DRIVER_FLAGS_FAVORITE;
    }
  }
}

void SaveAvailableGamesToProfile( void )
{
  int i;
  char numstring[10];

  PrfWriteProfileString( profile, "Quick List", NULL, NULL );
  // Nuke the previous list of available games

  for ( i=0; i < totaldrivers; ++i )
  {
    if ( availabledrivers[ i ] & DRIVER_FLAGS_AVAILABLE )
    {
      sprintf( numstring, "%d", i );

      PrfWriteProfileString( profile, "Quick List", numstring,
       drivers[i]->description );
    }
    if ( availabledrivers[ i ] & DRIVER_FLAGS_DANGEROUS )
      MarkGamePotentiallyDangerous( i );
  }
}

void SaveFavoriteGamesToProfile( void )
{
  int i;
  char numstring[10];

  PrfWriteProfileString( profile, "Favorite List", NULL, NULL );
  // toast the previous favorites list

  for ( i=0; i < totaldrivers; ++i )
  {
    if ( availabledrivers[ i ] & DRIVER_FLAGS_FAVORITE )
    {
      sprintf( numstring, "%d", i );

      PrfWriteProfileString( profile, "Favorite List", numstring,
       drivers[i]->description );
    }
  }
}

void MarkGamePotentiallyDangerous( int game )
{
  char numstring[10];

  sprintf( numstring, "%d", game );
  PrfWriteProfileString( profile, "Potentially Dangerous", numstring,
   drivers[game]->description );

  availabledrivers[ game ] |= DRIVER_FLAGS_DANGEROUS;
}

void ConfirmGameIsNotDangerous( int game )
{
  char numstring[10];

  if ( game < 0 )
  {
    myerrorlog( "Exitted game #-1??" );
    return;
  }

  sprintf( numstring, "%d", game );
  PrfWriteProfileString( profile, "Potentially Dangerous", numstring, NULL );

  availabledrivers[ game ] &= ~DRIVER_FLAGS_DANGEROUS;
}

char CheckIfGameIsDangerous( int game )
{
  return ( availabledrivers[ game ] & DRIVER_FLAGS_DANGEROUS );
}

void LoadFrameRateOptionsFromProfile( void )
{
  char data;
  ULONG datalen;

  data = 3;  datalen = 1;
  PrfQueryProfileData( profile, "Frame rate regulation", "Auto skip",
   &data, &datalen );
  globaloptions.autoskip = data;

  data = 0;  datalen = 1;
  PrfQueryProfileData( profile, "Frame rate regulation", "Manual skip",
   &data, &datalen );
  globaloptions.mskip = data;

  data = 1;  datalen = 1;
  PrfQueryProfileData( profile, "Frame rate regulation", "Use Timer0",
   &data, &datalen );
  globaloptions.usetimer0 = data;
}

void SaveFrameRateOptionsToProfile( void )
{
  PrfWriteProfileData( profile, "Frame rate regulation", "Auto skip",
   &globaloptions.autoskip, 1 );
  PrfWriteProfileData( profile, "Frame rate regulation", "Manual skip",
   &globaloptions.mskip, 1 );
  PrfWriteProfileData( profile, "Frame rate regulation", "Use Timer0",
   &globaloptions.usetimer0, 1 );
}

void LoadAudioOptionsFromProfile( void )
{
  char data;
  ULONG datalen;

  data = 3;  datalen = 1;
  PrfQueryProfileData( profile, "Audio settings", "Audio interleave",
   &data, &datalen );
  globaloptions.audinterleave = data;

  data = 5;  datalen = 1;
  PrfQueryProfileData( profile, "Audio settings", "Audio quality",
   &data, &datalen );
  switch ( data )
  {
    default:
    case 0:
      myerrorlog( "Bad audio quality value in INI file.  Defaulting to 22KHz." );
      // Intentional fall-through
    case 3:
      globaloptions.audfreq = 22050;
    break;
    case 1:
      globaloptions.audfreq = 8000;
    break;
    case 2:
      globaloptions.audfreq = 11025;
    break;
    case 4:
      globaloptions.audfreq = 33075;
    break;
    case 5:
      globaloptions.audfreq = 44100;
    break;
  }

  data = 1;  datalen = 1;
  PrfQueryProfileData( profile, "Audio settings", "Use DART",
   &data, &datalen );
  globaloptions.usedart = data;

  data = 1;  datalen = 1;
  PrfQueryProfileData( profile, "Audio settings", "Share audio",
   &data, &datalen );
  globaloptions.shareaudio = data;

  data = 1;  datalen = 1;
  PrfQueryProfileData( profile, "Audio settings", "Enable audio",
   &data, &datalen );
  globaloptions.enableaudio = data;
}

void SaveAudioOptionsToProfile( void )
{
  char audqualnum;

  switch ( globaloptions.audfreq )
  {
    case 8000:
      audqualnum = 1;
    break;
    case 11025:
      audqualnum = 2;
    break;
    case 22050:
      audqualnum = 3;
    break;
    case 33075:
      audqualnum = 4;
    break;
    case 44100:
      audqualnum = 5;
    break;
    default:
      myerrorlog( "Bad audio frequency setting.  Resetting to 22KHz." );
      audqualnum = 3;
    break;
  }

  PrfWriteProfileData( profile, "Audio settings", "Audio interleave",
   &globaloptions.audinterleave, 1 );
  PrfWriteProfileData( profile, "Audio settings", "Audio quality",
   &audqualnum, 1 );
  PrfWriteProfileData( profile, "Audio settings", "Use DART",
   &globaloptions.usedart, 1 );
  PrfWriteProfileData( profile, "Audio settings", "Share audio",
   &globaloptions.shareaudio, 1 );
  PrfWriteProfileData( profile, "Audio settings", "Enable audio",
   &globaloptions.enableaudio, 1 );
}

void DeallocVidModesIfNeeded( void )
{
  extern struct vidmode_descrip *MAME_modes, default_modes[];

  if ( MAME_modes == default_modes ) return;

  free( MAME_modes );

  MAME_modes = default_modes;
  MGLSC_clientState()->allowedmodes = MAME_modes;
}

void LoadMGLOptionsFromProfile( void )
{
  extern struct vidmode_descrip default_modes[], *MAME_modes;
  char data;
  USHORT usdata;
  ULONG datalen;
  int i, nummodes;
  char numstring[20];

  DeallocVidModesIfNeeded();

  data = 0;  datalen = 1;
  PrfQueryProfileData( profile, "MGL settings", "Number of modes defined",
   &data, &datalen );

  if ( !data )
  {
    MAME_modes = default_modes;
    MGLSC_clientState()->allowedmodes = MAME_modes;
    return;
  }

  nummodes = data;
  MAME_modes = (struct vidmode_descrip *)
   malloc( (nummodes + 1) * sizeof( struct vidmode_descrip ) );

  for ( i=0; i<nummodes; ++i )
  {
    sprintf( numstring, "Mode %d Width", i );
    usdata = 0;  datalen = 2;
    PrfQueryProfileData( profile, "MGL settings", numstring,
     &usdata, &datalen );
    MAME_modes[i].width = usdata;

    sprintf( numstring, "Mode %d Height", i );
    usdata = 0;  datalen = 2;
    PrfQueryProfileData( profile, "MGL settings", numstring,
     &usdata, &datalen );
    MAME_modes[i].height = usdata;

    sprintf( numstring, "Mode %d Depth", i );
    data = 0;  datalen = 1;
    PrfQueryProfileData( profile, "MGL settings", numstring,
     &data, &datalen );
    MAME_modes[i].depth = data;
  }

  MAME_modes[nummodes].width = 0;
  MAME_modes[nummodes].height = 0;
  MAME_modes[nummodes].depth = 0;

  MGLSC_clientState()->allowedmodes = MAME_modes;

  data = 1;  datalen = 1;
  PrfQueryProfileData( profile, "MGL settings", "Stretch FS video to screen size",
   &data, &datalen );
  globaloptions.stretchblit = data;
  MGLSC_clientState()->stretchblit = data;

  data = 0;  datalen = 1;
  PrfQueryProfileData( profile, "MGL settings", "Allow custom mode creation",
   &data, &datalen );
  globaloptions.customfs = data;
  MGLSC_clientState()->custommodes = data;
}

void SaveMGLOptionsToProfile( void )
{
  extern struct vidmode_descrip default_modes[], *MAME_modes;
  char data;
  USHORT usdata;
  int i;
  char numstring[20];

  data = globaloptions.stretchblit;
  PrfWriteProfileData( profile, "MGL settings", "Stretch FS video to screen size",
   &data, 1 );
  data = globaloptions.customfs;
  PrfWriteProfileData( profile, "MGL settings", "Allow custom mode creation",
   &data, 1 );

  if ( MAME_modes == default_modes ) return;
  // Nothing different from defaults, so don't bother

  for ( i=0; MAME_modes[i].width; ++i )
  {
    sprintf( numstring, "Mode %d Width", i );
    usdata = MAME_modes[i].width;
    PrfWriteProfileData( profile, "MGL settings", numstring,
     &usdata, 2 );

    sprintf( numstring, "Mode %d Height", i );
    usdata = MAME_modes[i].height;
    PrfWriteProfileData( profile, "MGL settings", numstring,
     &usdata, 2 );

    sprintf( numstring, "Mode %d Depth", i );
    data = MAME_modes[i].depth;
    PrfWriteProfileData( profile, "MGL settings", numstring,
     &data, 1 );
  }

  data = i;
  PrfWriteProfileData( profile, "MGL settings", "Number of modes defined",
   &data, 1 );
}

int WasSplashScreenShown( void )
{
  USHORT data;
  ULONG datalen = sizeof( USHORT );

  PrfQueryProfileData( profile, "Dedication screen", "Previously viewed",
   &data, &datalen );

  if ( data == 911 ) return 1;

  return 0;
}

void SplashScreenWasShown( void )
{
  USHORT data = 911;

  PrfWriteProfileData( profile, "Dedication screen", "Previously viewed",
   &data, sizeof( USHORT ) );
}

void LoadMiscOptionsFromProfile( void )
{
  USHORT data = 1;
  ULONG datalen = 1;

  PrfQueryProfileData( profile, "Miscellaneous options", "Pause on window focus change",
   &data, &datalen );
  globaloptions.autopause = data;

  data = 1; datalen = 1;
  PrfQueryProfileData( profile, "Miscellaneous options", "Simulate coin slot lights with keyboard LEDs",
   &data, &datalen );
  globaloptions.simled = 1;
}

void SaveMiscOptionsToProfile( void )
{
  PrfWriteProfileData( profile, "Miscellaneous options", "Pause on window focus change",
   &globaloptions.autopause, 1 );
  PrfWriteProfileData( profile, "Miscellaneous options", "Simulate coin slot lights with keyboard LEDs",
   &globaloptions.simled, 1 );
}

void LoadRapidKeysForGame( int game )
{
  extern char rapidkeys[];               // in os2.c

  unsigned char data, numkeys, i;
  char str[40];
  ULONG datalen;
  char *gamename;

  memset( rapidkeys, 0, 256 );

  if ( game == -1 )
  {
    gamename = "Rapid fire defaults";
  } else {
    gamename = (char *)drivers[ game ]->name;
  }

  data = 0; datalen = 1;
  PrfQueryProfileData( profile, "Rapid fire rate", gamename,
   &data, &datalen );

  if ( !data )
  {
    // No specific info for this game, so revert to defaults

    gamename = "Rapid fire defaults";
    data = 10; datalen = 1;
    PrfQueryProfileData( profile, "Rapid fire rate", gamename,
     &data, &datalen );
  }

  globaloptions.rapidrate = data;

  numkeys = 0; datalen = 1;
  PrfQueryProfileData( profile, "Rapid fire key number", gamename,
   &numkeys, &datalen );

  for ( i=0; i<numkeys; ++i )
  {
    sprintf( str, "Rapid fire key #%d", i + 1 );
    data = 255; datalen = 1;
    PrfQueryProfileData( profile, str, gamename, &data, &datalen );
    if ( data != 255 )
    {
      rapidkeys[ data ] = 1;
    }
  }
}

void SaveRapidKeysForGame( int game )
{
  extern char rapidkeys[];  // in os2.c

  int i;
  char numrapid = 0, *gamename;
  char str[40];

  if ( game == -1 )
  {
    gamename = "Rapid fire defaults";
  } else {
    gamename = (char *)drivers[ game ]->name;
  }

  PrfWriteProfileData( profile, "Rapid fire rate", gamename,
   &globaloptions.rapidrate, 1 );

  for ( i=0; i<256; ++i )
  {
    if ( rapidkeys[i] )
    {
      sprintf( str, "Rapid fire key #%d", numrapid + 1 );

      PrfWriteProfileData( profile, str, gamename, &i, 1 );
      numrapid++;
    }
  }

  PrfWriteProfileData( profile, "Rapid fire key number", gamename,
   &numrapid, 1 );
}

void LoadInputOptionsFromProfile( int game )
{
  ULONG datalen;
  char *gamename, data;

  if ( game == -1 )
  {
    gamename = "Input options defaults";
  } else {
    gamename = (char *)drivers[ game ]->name;
  }

  data = 1;  datalen = 1;
  PrfQueryProfileData( profile, "Enable mouse input", gamename,
   &data, &datalen );
  globaloptions.enablemouse = data;

  data = 0;  datalen = 1;
  PrfQueryProfileData( profile, "Grab mouse control in PM", gamename,
   &data, &datalen );
  globaloptions.mousegrab = data;

  data = 1;  datalen = 1;
  PrfQueryProfileData( profile, "Bypass PM keyboard handling when game is running", gamename,
   &data, &datalen );
  globaloptions.altbypass = data;

}

void SaveInputOptionsToProfile( int game )
{
  char *gamename;

  if ( game == -1 )
  {
    gamename = "Input options defaults";
  } else {
    gamename = (char *)drivers[ game ]->name;
  }

  PrfWriteProfileData( profile, "Enable mouse input", gamename,
   &globaloptions.enablemouse, 1 );
  PrfWriteProfileData( profile, "Grab mouse control in PM", gamename,
   &globaloptions.mousegrab, 1 );
  PrfWriteProfileData( profile, "Bypass PM keyboard handling when game is running",
   gamename, &globaloptions.altbypass, 1 );
}

